
/* as296_main.c
 *
 * Date: 31.03.06
 * Version: 0.1.5
 * Programmer: DG1CPA@DB0CHZ.#SAX.DEU.EU
 *
 * Ham Radio Modem driver
 * for AS296 USB-Modem from AAtis (9600bd FSK, 4800bd FSK, 1200bd AFSK)
 *
 * This driver connects a AS296 USB Modem to a tty (pseudo-tty)
 * from the other side of the pseudo-tty you can connect a kiss-driver (kissattach)
 * The kissdriver has to use the "rmnc-crc" protocol
 *
 * This program is distributed under the GPL, version 2.
 */


#include <usb.h>
#include <string.h>
#include <stdio.h>
#include "as296_lib.h"

#define as296_error_return(code, str) do {  \
        as296->error_str = str;             \
        return code;                       \
   } while(0);                 


/* as296_init

  Initializes a as296_context.

  Return codes:
   0: All fine
  -1: Couldn't allocate read buffer
*/
int as296_init(struct as296_context *as296)
{
    as296->usb_dev = NULL;
    as296->usb_read_timeout = 5000; //5000==5sekunden
    as296->usb_write_timeout = 0;

    as296->interface = 0;

    //#, Endaddressen fuer read/write
    as296->in_ep = 0x81;
    as296->out_ep = 0x02;

    as296->as296_mode = 0;
    as296->as296_txdelay = 0;
    
    as296->error_str = NULL;

    /* All fine. Now allocate the readbuffer */
    return 1; 
}

/* as296_set_usbdev
 
   Use an already open device.
*/
void as296_set_usbdev (struct as296_context *as296, usb_dev_handle *usb)
{
    as296->usb_dev = usb;
}


/* as296_usb_find_all
 
   Finds all as296 devices on the usb bus. Creates a new as296_device_list which
   needs to be deallocated by as296_list_free after use.

   Return codes:
    >0: number of devices found
    -1: usb_find_busses() failed
    -2: usb_find_devices() failed
    -3: out of memory
*/
int as296_usb_find_all(struct as296_context *as296, struct as296_device_list **devlist, int vendor, int product) 
{
    struct as296_device_list **curdev;
    struct usb_bus *bus;
    struct usb_device *dev;
    int count = 0;
    
    usb_init();
    if (usb_find_busses() < 0)
        as296_error_return(-1, "usb_find_busses() failed");
    if (usb_find_devices() < 0)
        as296_error_return(-2, "usb_find_devices() failed");

    curdev = devlist;
    for (bus = usb_busses; bus; bus = bus->next) {
        for (dev = bus->devices; dev; dev = dev->next) {
            if (dev->descriptor.idVendor == vendor
                    && dev->descriptor.idProduct == product)
            {
                *curdev = (struct as296_device_list*)malloc(sizeof(struct as296_device_list));
                if (!*curdev)
                    as296_error_return(-3, "out of memory");
                
                (*curdev)->next = NULL;
                (*curdev)->dev = dev;

                curdev = &(*curdev)->next;
                count++;
            }
        }
    }
    
    return count;
}

/* as296_list_free

   Frees a created device list.
*/
void as296_list_free(struct as296_device_list **devlist) 
{
    struct as296_device_list **curdev;
    for (; *devlist == NULL; devlist = curdev) {
        curdev = &(*devlist)->next;
        free(*devlist);
    }

    devlist = NULL;
}

/* as296_usb_open_dev 

   Opens a as296 device given by a usb_device.
   
   Return codes:
     0: all fine
    -4: unable to open device
    -5: unable to claim device
    -6: reset failed
    -7: set baudrate failed
*/
int as296_usb_open_dev(struct as296_context *as296, struct usb_device *dev)
{
    int test=0;
    
    if (!(as296->usb_dev = usb_open(dev)))
        as296_error_return(-4, "usb_open() failed");
#if defined(LIBUSB_HAS_DETACH_KERNEL_DRIVER_NP)   
    if(0 != (test = usb_detach_kernel_driver_np(as296->usb_dev,0))) //remove usbhid.ko WE ARE NO KEYBOARD!!!
    {
//        printf("%d remove kerneldriver 2nd times?\n",test);
    }
#endif
    
    if (usb_claim_interface(as296->usb_dev,as296->interface) != 0) {
        usb_close (as296->usb_dev);
        as296_error_return(-5, "unable to claim usb device.");
    }

    

    if (as296_set_baudrate (as296) != 0) {  // baudrate is to set in the as296 structur
        usb_close (as296->usb_dev);
        as296_error_return(-7, "set baudrate failed");
    }

    as296_error_return(0, "all fine");
}

/* as296_usb_open
   
   Opens the first device with a given vendor and product ids.
   
   Return codes:
   See as296_usb_open_desc()
*/  
int as296_usb_open(struct as296_context *as296, int vendor, int product)
{
    return as296_usb_open_desc(as296, vendor, product, NULL, NULL);
}

/* as296_usb_open_desc

   Opens the first device with a given, vendor id, product id,
   description and serial.
   
   Return codes:
     0: all fine
    -1: usb_find_busses() failed
    -2: usb_find_devices() failed
    -3: usb device not found
    -4: unable to open device
    -5: unable to claim device
    -6: reset failed
    -7: set baudrate failed
    -8: get product description failed
    -9: get serial number failed
    -10: unable to close device
*/
int as296_usb_open_desc(struct as296_context *as296, int vendor, int product,
                       const char* description, const char* serial)
{
    struct usb_bus *bus;
    struct usb_device *dev;
    char string[256];

    usb_init();

    if (usb_find_busses() < 0)
        as296_error_return(-1, "usb_find_busses() failed");
    if (usb_find_devices() < 0)
        as296_error_return(-2, "usb_find_devices() failed");

    for (bus = usb_busses; bus; bus = bus->next) {
        for (dev = bus->devices; dev; dev = dev->next) {
            if (dev->descriptor.idVendor == vendor
                    && dev->descriptor.idProduct == product) {
                if (!(as296->usb_dev = usb_open(dev)))
                    as296_error_return(-4, "usb_open() failed");

                if (description != NULL) {
                    if (usb_get_string_simple(as296->usb_dev, dev->descriptor.iProduct, string, sizeof(string)) <= 0) {
                        usb_close (as296->usb_dev);
                        as296_error_return(-8, "unable to fetch product description");
                    }
                    if (strncmp(string, description, sizeof(string)) != 0) {
                        if (usb_close (as296->usb_dev) != 0)
                            as296_error_return(-10, "unable to close device");
                        continue;
                    }
                }
                if (serial != NULL) {
                    if (usb_get_string_simple(as296->usb_dev, dev->descriptor.iSerialNumber, string, sizeof(string)) <= 0) {
                        usb_close (as296->usb_dev);
                        as296_error_return(-9, "unable to fetch serial number");
                    }
                    if (strncmp(string, serial, sizeof(string)) != 0) {
                        if (usb_close (as296->usb_dev) != 0)
                            as296_error_return(-10, "unable to close device");
                        continue;
                    }
                }

                if (usb_close (as296->usb_dev) != 0)
                    as296_error_return(-10, "unable to close device");
                
                return as296_usb_open_dev(as296, dev);
            }
        }
    }

    // device not found
    as296_error_return(-3, "device not found");
}

/* as296_usb_close
   
   Closes the as296 device.
   
   Return codes:
     0: all fine
    -1: usb_release failed
    -2: usb_close failed
*/
int as296_usb_close(struct as296_context *as296)
{
    int rtn = 0;

    if (usb_release_interface(as296->usb_dev, as296->interface) != 0)
        rtn = -1;

    if (usb_close (as296->usb_dev) != 0)
        rtn = -2;

    return rtn;
}


/*
    as296_set_baudrate - Sets channel baudrate and mode #todo
    
    Return codes:
     0: all fine
    -1: invalid baudrate
    -2: setting baudrate failed
*/
int as296_set_baudrate(struct as296_context *as296)
{
    unsigned char as296_config[7]={0};

    as296_config[0]=0x10|as296->as296_mode;  //0y10|Mode 0=9k6 3=1k2 ...
    as296_config[1]=as296->as296_txdelay;    //txd96 jeweils durch 10
    as296_config[2]=as296->as296_txdelay;    //txd12
    as296_config[3]=2;                       //rxd96
    as296_config[4]=2;                       //rxd12
    as296_config[5]=as296->as296_txdelay;    //txd48
    as296_config[6]=2;                       //rxd48
    as296_config[7]=0xC0;                    //FEND

    int ret = usb_control_msg(
                as296->usb_dev,                                  // usb_dev_handle *dev
                USB_TYPE_CLASS|USB_RECIP_INTERFACE,             //0x00000021,                                     // int requesttype c8 (0x01<<5 TYPE_CLASS)|(0x01 RECIP_INTERFACE)
                USB_REQ_SET_CONFIGURATION,                      //0x00000009,                                     // int request 12 (9=set_config)
                0x0300,                                              // int value
                0,                                              // int index
                (char *) as296_config,                          // char *bytes
                0x00000008,                                     // int size
                5000                                            // int timeout
        );

    if(ret != 8) as296_error_return (-2, "Setting new config/baudrate failed");
    
//    if (ret == 8) printf("Baudrate/Mode ist set\n");
    return 0;
}


int as296_write_data(struct as296_context *as296, char *buf, int size)
{
    int ret;

    ret = usb_interrupt_write(as296->usb_dev, as296->out_ep, buf, size, as296->usb_write_timeout);
    if (ret < 0)
        as296_error_return(ret, "usb bulk write failed");

    return ret;
}

int as296_read_data(struct as296_context *as296, char *buf, int size)
{
    int ret;

        /* returns how much received */
//        printf("usbdev=%X in_ep=%X buf=%X size=%d timeout=%d\n",(int)as296->usb_dev, as296->in_ep, (int)buf, size, as296->usb_read_timeout);
        ret = usb_interrupt_read (as296->usb_dev, as296->in_ep, buf, size, as296->usb_read_timeout);

        if (ret < 0)
            as296_error_return(ret, "usb bulk read failed");

        return ret;
}


char *as296_get_error_string (struct as296_context *as296)
{
    return as296->error_str;
}
