#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2011 by Michael Conrad <do5mc at aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

include APRS4R


class StartupStatusHandler

  @@logger = APRS4RLogger.get_logger( "StartupStatusHandler")


  def StartupStatusHandler.header
    return "Status - Start"
  end


  def StartupStatusHandler.cgi_parameter( cgi, name)
    value = nil

    parameter = cgi.params[name]
    
    if parameter 
      value = parameter[0]
    end

    return value
  end


  def StartupStatusHandler.handle_request( cgi)
    @@logger.info( "handle_request( cgi)")

    etc_dir = ENV['APRS4R_ETC'] || "/etc/aprs4r"
    @@logger.debug( "etc_dir: #{etc_dir}")

    default_file = ENV['APRS4R_DEFAULT'] || "/etc/default/aprs4r"
    @@logger.debug( "default_file: #{default_file}")

    daemon_file = ENV['APRS4R_DAEMON'] || "/usr/bin/aprs4r"
    @@logger.debug( "daemon_file: #{daemon_file}")

    daemon_pid_file = ENV['APRS4R_PID'] || "/var/run/aprs4r.pid"
    @@logger.debug( "daemon_pid_file: #{daemon_pid_file}")

    watchdog_file = ENV['APRS4R_WATCHDOG'] || "/usr/bin/aprs4r_watchdog"
    @@logger.debug( "watchdog_file: #{watchdog_file}")

    watchdog_pid_file = ENV['APRS4R_WATCHDOG_PID'] || "/var/run/aprs4r_watchdog.pid"
    @@logger.debug( "watchdog_pid_file: #{watchdog_pid_file}")

    script_file = ENV['APRS4R_WEB_SCRIPT'] || "/usr/bin/aprs4r-web-script"
    @@logger.debug( "script_file: #{script_file}")


    # list config files
    config_files = Array.new
    Dir.new( etc_dir).sort.each { |name|
      if name =~ /\.yaml$/ && name != "default.yaml"
        config_files << name
      end
    }

    
    action = "unknown"
    action = "start_daemon" if cgi_parameter( cgi, 'start_daemon')
    action = "stop_daemon" if cgi_parameter( cgi, 'stop_daemon')
    action = "start_watchdog" if cgi_parameter( cgi, 'start_watchdog')
    action = "stop_watchdog" if cgi_parameter( cgi, 'stop_watchdog')
    action = "save" if cgi_parameter( cgi, 'save')

    result = nil
    
    @@logger.debug( "action: #{action}")
    
    if action == "start_daemon"
      @@logger.debug( "start daemon command")

      file = cgi_parameter( cgi, 'file')
      config_file = nil
      config_file = File.basename( file) if file

      @@logger.debug( "system: #{script_file} #{daemon_file} #{etc_dir}/#{config_file}")
      system( "#{script_file} #{daemon_file} #{etc_dir}/#{config_file}")
      result = $?

      sleep 5
    elsif action == "stop_daemon" 
      @@logger.debug( "stop daemon command")
      
      # read out pid file
      daemon_pid = -1
      
      begin 
        File.open( daemon_pid_file).each { |line|
          daemon_pid = line.chomp!.to_i if line && line =~ /^[0-9]*$/
        }
      rescue Exception
      end
      
      @@logger.debug( "daemon_pid: #{daemon_pid}")

      # send kill signal 
      if daemon_pid > 0 
        begin
          Process.kill( 9, daemon_pid)
        rescue Exception => ex
          @@logger.error( "Error sending KILL signal to daemon_pid: #{daemon_pid}, ex: #{ex}")
        end
      end
      
      sleep 5
      
    elsif action == "start_watchdog"
      @@logger.debug( "start watchdog command")
      
      @@logger.debug( "system: #{script_file} #{watchdog_file}")
      system( "#{script_file} #{watchdog_file}")
      result = $?

      sleep 5

    elsif action == "stop_watchdog"
      @@logger.debug( "stop watchdog command")
      
      # read out pid file
      watchdog_pid = -1
      
      begin 
        File.open( watchdog_pid_file).each { |line|
          watchdog_pid = line.chomp!.to_i if line && line =~ /^[0-9]*$/
        }
      rescue Exception
      end
      
      @@logger.debug( "watchdog_pid: #{watchdog_pid}")
      
      # send kill signal 
      if watchdog_pid > 0 
        begin
          Process.kill( 9, watchdog_pid)
        rescue Exception => ex
          @@logger.error( "Error sending KILL signal to watchdog_pid: #{watchdog_pid}, ex: #{ex}")
        end
      end

      sleep 5

    elsif action == "save"
      @@logger.debug( "save command")


      daemon = cgi_parameter( cgi, 'daemon')
      file = cgi_parameter( cgi, 'file')
      config_file = File.basename( file) if file
      memory = cgi_parameter( cgi, 'memory')
      
      @@logger.debug( "daemon: #{daemon}")
      @@logger.debug( "file: #{file}")
      @@logger.debug( "config_file: #{config_file}")
      @@logger.debug( "memory: #{memory}")

      watchdog = cgi_parameter( cgi, 'watchdog')
      timeout = cgi_parameter( cgi, 'timeout')

      @@logger.debug( "watchdog: #{watchdog}")
      @@logger.debug( "timeout: #{timeout}")

      if config_files.include?( config_file)
        begin
          File.open( default_file, "w") { |file|
            file.puts( "#")
            file.puts( "# aprs4r default file")
            file.puts( "#")
            file.puts

            if daemon == "true"
              file.puts( "START_DAEMON=\"true\"")
            else
              file.puts( "START_DAEMON=\"false\"")
            end
            
            file.puts
            file.puts( "PROFILE=\"#{config_file}\"")
            file.puts
            file.puts( "ULIMIT=\"#{memory}\"")
            file.puts

            if watchdog == "true"
              file.puts( "START_WATCHDOG=\"true\"")
            else
              file.puts( "START_WATCHDOG=\"false\"")
            end

            file.puts
            file.puts( "TIMEOUT=\"#{timeout}\"")
            file.puts

          }

        rescue Exception => ex
          @@logger.error( "Error writing default file #{default_file}")
        end
      end
    end


    # auto start 
    daemon_options = [ [ "true", "ja"], ["false", "nein", true] ]
    start_daemon = false
    begin
      File.new( default_file).each{ |line|
        if line =~ /START_DAEMON/ && (line =~ /true/ || line =~ /TRUE/)
          start_daemon = true
          daemon_options = [ [ "true", "ja", true], ["false", "nein"] ]
        end
      }
    rescue Exception
    end
    
    # start profile
    start_profile = ""
    begin
      File.new( default_file).each{ |line|
        if line =~ /PROFILE/
          start_profile = line.chomp!.split( /\=/)[1]
          start_profile.gsub!( /\"/, "")
        end
      }
    rescue Exception
    end


    profile_options = []
    config_files.each{ |config_file|
      @@logger.debug( "config_file: #{config_file}")
      if config_file == start_profile
        profile_options << [config_file, config_file, true]
      else 
        profile_options << [config_file, config_file]
      end
    }


    # memory
    memory_default = true
    memory_options = [ ["16384", "16MB"], ["20480", "20MB"], ["24576", "24MB"], ["28672", "28MB"], ["32768", "32MB"], ["65536", "64MB"] ]
    begin
      File.new( default_file).each{ |line|
        if line =~ /ULIMIT="(.*)"/
          memory_options.each { |option|
            if option[0] == $1 && option.length == 2
              option << true
              memory_default = false
            end
          }
        end
      }
    rescue Exception
    end

    # if no limit is set, use 24MB as default
    if memory_default == true
      memory_options[2] << true
    end


    # watchdog
    watchdog_options = [ [ "true", "ja"], ["false", "nein", true] ]
    start_watchdog = false
    begin
      File.new( default_file).each{ |line|
        if line =~ /START_WATCHDOG/ && (line =~ /true/ || line =~ /TRUE/)
          start_watchdog = true
          watchdog_options = [ [ "true", "ja", true], ["false", "nein"] ]
        end
      }
    rescue Exception
    end
    
    # timeout
    timeout = ""
    begin
      File.new( default_file).each{ |line|
        if line =~ /TIMEOUT/
          timeout = line.chomp!.split( /\=/)[1]
          timeout.gsub!( /\"/, "")
        end
      }
    rescue Exception
    end



    # read out daemon_pid file
    daemon_pid = -1
    
    begin 
      File.open( daemon_pid_file).each { |line|
        @@logger.debug( "daemon_pid: line: #{line}")
        daemon_pid = line.chomp!.to_i if line && line =~ /^[0-9]*$/
      }
    rescue Exception => ex
      @@logger.warn( "daemon_pid_file: #{ex}")
    end
    
    @@logger.debug( "daemon_pid: #{daemon_pid}")
    # process check
    daemon_pid_result = -1
    
    if daemon_pid > 0 
      begin
        daemon_pid_result = Process.kill( 0, daemon_pid)
      rescue Exception
      end
    end

    daemon_pid_status = "nicht aktiv"
    if daemon_pid_result != -1 
      daemon_pid_status = "aktiv (PID #{daemon_pid})"
    end


    # read out watchdog_pid file
    watchdog_pid = -1
    
    begin 
      File.open( watchdog_pid_file).each { |line|
        watchdog_pid = line.chomp!.to_i if line && line =~ /^[0-9]*$/
      }
    rescue Exception
    end
    
    @@logger.debug( "watchdog_pid: #{watchdog_pid}")
    # process check
    watchdog_pid_result = -1
    
    if watchdog_pid > 0 
      begin
        watchdog_pid_result = Process.kill( 0, watchdog_pid)
      rescue Exception
      end
    end

    watchdog_pid_status = "nicht aktiv"
    if watchdog_pid_result != -1 
      watchdog_pid_status = "aktiv (PID #{watchdog_pid})"
    end


    output = "" 

    # aprs4r start/stop
    output += cgi.h2 { "APRS4R manuell Starten/Stoppen" } 

    output += cgi.table { 
      cgi.form( "ACTION" => "/aprs4r-web/cgi-bin/admin.rb/status/startup") { 

        cgi.tr { 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { "APRS4R-Software:" } 
          } + 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { daemon_pid_status } 
          } + 
          cgi.td { }
        } + 

        cgi.tr { 
          cgi.td { cgi.big { "Aktion: " } } + 
          cgi.td { 
            form_output = "" 

            if daemon_pid_result == -1 
              form_output = cgi.popup_menu( "NAME" => "file", "VALUES" => profile_options) 
            end

            form_output
          } + 
          cgi.td { 
            form_output = "" 

            if daemon_pid_result == -1 
              form_output = cgi.submit( "VALUE" => "Starten", "NAME" => "start_daemon")
            else 
              form_output = cgi.submit( "VALUE" => "Stoppen", "NAME" => "stop_daemon")
            end
            
            form_output
          }
        } 
      }
    }

      
    # watchdog  start/stop
    output += cgi.h2 { "Watchdog manuell Starten/Stoppen" } 

    output += cgi.table { 

      cgi.form( "ACTION" => "/aprs4r-web/cgi-bin/admin.rb/status/startup") { 

        cgi.tr { 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { "APRS4R-Watchdog:" } 
          } + 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { watchdog_pid_status } 
          } + 
          cgi.td { }
        } +
          
        cgi.tr { 
          cgi.td { cgi.big { "Aktion: " } } + 
          cgi.td { } + 
          cgi.td { 
            form_output = "" 

            if watchdog_pid_result == -1 
              form_output = cgi.submit( "VALUE" => "Starten", "NAME" => "start_watchdog")
            else 
              form_output = cgi.submit( "VALUE" => "Stoppen", "NAME" => "stop_watchdog")
            end
            
            form_output
          }
        }
      }
    }
    

    # automatic start at boot
    output += cgi.h2 { "Automatischer Start" } 

    output += cgi.table { 
      cgi.form( "ACTION" => "/aprs4r-web/cgi-bin/admin.rb/status/startup") { 

        cgi.tr { 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { "Start beim Booten:" } 
          } + 
          cgi.td( "WIDTH" => "200") { 
            cgi.popup_menu( "NAME" => "daemon", "VALUES" => daemon_options)
          } + 
          cgi.td { }
        } + 

        cgi.tr { 
          cgi.td { cgi.big { "Startprofil:" } } + 
          cgi.td { 
            cgi.popup_menu( "NAME" => "file", "VALUES" => profile_options) 
          } + 
          cgi.td { }
        } +

        cgi.tr { 
          cgi.td { cgi.big { "Speicherlimit:" } } + 
          cgi.td { 
            cgi.popup_menu( "NAME" => "memory", "VALUES" => memory_options) 
          } + 
          cgi.td { }
        } +

        cgi.tr { 
          cgi.td( "WIDTH" => "200") { 
            cgi.big { "Watchdog:" } 
          } + 
          cgi.td( "WIDTH" => "200") { 
            cgi.popup_menu( "NAME" => "watchdog", "VALUES" => watchdog_options)
          } + 
          cgi.td { }
        } +

        cgi.tr { 
          cgi.td { cgi.big { "Timeout:" } } + 
          cgi.td { 
            cgi.text_field( "timeout", timeout, 6) 
          } + 
          cgi.td { 
            cgi.submit( "Speichern", "save") 
          }
        }          
      }
    }

    
    return output
  end

end

