#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2010 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'thread'
require 'socket'
require 'timeout'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/Plugin'

require 'aprs4r/server/T2ServerSocket'
require 'aprs4r/server/T2ServerPluginConfiguration'


module APRS4R

  class T2ServerPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "T2ServerPlugin")


    def initialize
      logger.info( "initialize")

      super

      @server_thread_run = false
      @server_thread = nil

      @listen_thread_run = false
      @listen_thread = nil

      @server_queue = Queue.new

      @clients = Array.new

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @inDevice = configuration.inDevice
      @outDevice = configuration.outDevice

      @port = 14580
      @port = configuration.port.to_i if configuration.port

      return
    end

    
    def start
      logger.info( "start")

      return if !@enable

      @server_socket = T2ServerSocket.new( @port)

      @server_thread_run = true
      @server_thread = Thread.new{ server_thread_method }

      @listen_thread_run = true
      @listen_thread = Thread.new{ listen_thread_method }

      register_listener

      return
    end

    
    def stop
      logger.info( "stop")

      return if ! @enable

      unregister_listener

      # TODO stop threads

      return
    end

    
    def register_listener
      logger.info( "register_listener")

      @socket_manager.add_listener( @inDevice, self)

      return
    end


    def unregister_listener
      logger.info( "register_listener")

      @socket_manager.remove_listener( @inDevice, self)

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      if message.nil?
        logger.warn( "message: nil")
        return
      end

      # store message in queue and return immediately
      @server_queue.push( message)
    
    end


    def server_thread_method
      logger.info( "server_thread_method")


      while @server_thread_run

        message = @server_queue.pop
        logger.debug( "received message: #{message}")

        @clients.each{ |client|
          begin
            logger.debug( "write to client: #{client}")
            client.write_message( message)
          rescue Exception => ex
            logger.warn( "write_message: ex: #{ex}")
            @clients.delete( client)
          end
        }
      end

    end


    def listen_thread_method
      logger.info( "listen_thread_method")

      while @listen_thread_run 

        socket = @server_socket.accept
        clientSocket = nil
        logger.debug( "socket: #{socket}")

        begin
          Timeout::timeout( 5) do 
            clientSocket = @server_socket.setup( socket, @server_queue)
          end
        rescue Exception => ex
          logger.warn( "socket setup failed: #{ex}")
          clientSocket = nil
        end

        if clientSocket 
          logger.debug( "adding client: #{clientSocket}")
          @clients << clientSocket
        end
        
      end

    end


  end

end
