#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'timeout'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/WS2300WeatherPluginConfiguration'


module APRS4R
  
  class WS2300WeatherPlugin < WeatherPlugin
    
    @logger = APRS4RLogger.get_logger( "WS2300WeatherPlugin")
    
    
    @@WIND_DIRECTION_INVALID = "0.0"
    @@WIND_SPEED_INVALID = "51.0"
    
    @@TEMPERATURE_INVALID = "81.1"
    
    @@HUMIDITY_INVALID = "110"
    
    
    def initialize
      logger.info( "initialize")
      
      super
      
      return
    end
    
    
    def setup( configuration)
      logger.info( "setup( configuration)")
      
      super( configuration)
      
      @command = configuration.command
      @output = "/proc/self/fd/1"
      @config = configuration.config
      
      @period = configuration.period
      @offset = configuration.offset
      @timeout = configuration.timeout
      
      return
    end
    

    def start 
      logger.info( "start")

      if !File.file?( @command)
        logger.error( "binary #{@command} not found")
        return
      end

      if !File.file?( @config)
        logger.error( "config #{@config} not found")
        return
      end

      if !File.executable?( @command)
        logger.error( "binary #{@command} not an executable")
        return
      end

      Thread.new { runWS2300Thread }
      
      return
    end

    
    def runWS2300Thread
      logger.info( "runWS2300Thread")
      
      sleep @offset
      
      while true 

        begin
          
          readWeatherData
          
          sleep @period
          
        rescue Exception => ex
          logger.warn( "runWS2300Thread: ex: #{ex}")
        end
        
      end
      
      return
    end
    


    def readWeatherData
      logger.info( "readWeatherData")
      
      data = readData
      
      if data
        values = data.split( / /)
        
        @wind_direction = nil
        @wind_direction = values[9].to_i if values[9] && values[9] != @@WIND_DIRECTION_INVALID
        
        @wind_avg = nil
        @wind_avg = values[8].to_i if values[8] && values[8] != @@WIND_SPEED_INVALID
        
        @wind_gust = nil
        
        
        @rain_1h = nil
        @rain_1h = values[12].to_i if values[12]
        
        @rain_total = nil
        @rain_total = values[14].to_i if values[14]
        
        @rain_today = nil
        @rain_today = values[13].to_i if values[13]
        
        
        @temperature = nil
        @temperature = values[4].to_i if values[4] && values[4] != @@TEMPERATURE_INVALID
        
        @humidity = nil
        @humidity = values[7].to_i if values[7] && values[7] != @@HUMIDITY_INVALID
        
        @dewpoint = nil
        @dewpoint = values[5].to_i if values[5]
        
        @pressure = nil
        @pressure = values[15].to_f if values[15]

        update_timestamp
      else
        logger.warn( "no data received")
      end
    
      return
    end


    def readData
      logger.info( "readData")

      # return "20100310203949 2010-Mar-10 20:39:49 25.9 1.5 -7.8 19 50 51.0 0.0 N 81.1 0.00 0.00 2402.48 6656.800 Falling Rainy"

      data = nil

      begin

        Timeout::timeout( @timeout) do

          data = ` "#{@command}" "#{@output}" "#{@config}" `

          return nil if $?.to_i != 0
          
        end

      rescue Exception => ex
        logger.warn( "ws2300: #{ex}")
      end

      return data
    end

  end

end
