#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2009 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'io/nonblock'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/UltimeterWeatherPluginConfiguration'
require 'aprs4r/UltimeterWeatherMessage'


module APRS4R

  class UltimeterWeatherPlugin < WeatherPlugin

    @logger = APRS4RLogger.get_logger( "UltimeterWeatherPlugin")


    def initialize
      logger.info( "initialize")

      super

      return
    end
    

    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @file = configuration.file

      init_serial

      return
    end


    def init_serial
      logger.info( "init_serial")

      begin 
        @port = SerialPort.new( @file, 2400, 8, 1, SerialPort::NONE)
        @port.nonblock = true 
        @port.sync = true
      rescue Exception => ex
        logger.warn( "init_serial: ex: #{ex}")
      end

      return
    end


    def reinit_serial
      logger.info( "reinit_serial")

      begin
        Timeout::timeout( 5) do
          @port.close if @port
        end
      rescue Exception => ex
        logger.warn( "reinit_serial: close: #{ex}")
      end

      init_serial
      
      return
    end


    def start
      logger.info( "start")

      Thread.new { runUltimeterThread }

      return
    end


    def runUltimeterThread
      logger.info( "runUltimeterThread")
      
      while true 

        begin
          readWeatherData
        rescue Exception => ex
          logger.warn( "runUltimeterThread: ex: #{ex}")
        end

      end

      return
    end


    def readWeatherData
      logger.info( "readWeatherData")

      record = readRecord

      if record 
        if record.has_wind?
          logger.info( "wind: direction: #{record.wind_direction}, avg: #{record.wind_avg}, gust: #{record.wind_gust}")
          @wind_direction = record.wind_direction
          @wind_avg = record.wind_avg
          @wind_gust = record.wind_gust
          update_timestamp
        end
        
        if record.has_rain?
          logger.info( "rain: 1h: #{record.rain_1h}, 24h: #{record.rain_24h}, today: #{record.rain_today}")
          @rain_1h = record.rain_1h
          @rain_total = record.rain_24h
          @rain_today = record.rain_today
          update_timestamp
        end
        
        if record.has_thermo_outdoor?
          logger.info( "thermo: temperature: #{record.temperature}, humidity: #{record.humidity}, dewpoint: #{record.dewpoint}")
          @temperature = record.temperature
          @humidity = record.humidity
          @dewpoint = record.dewpoint
          update_timestamp
        end

        if record.has_baro?
          logger.info( "baro: pressure: #{record.pressure_relative}")
          @pressure = record.pressure_relative
          update_timestamp
        end
      else 
        logger.warn( "empty record received")
      end
      
      return
    end


    def readRecord
      logger.info( "readRecord")

      payload = nil

      begin
        Timeout::timeout( 60) do
          payload = @port.gets if @port 
        end
      rescue Exception => ex
        logger.warn( "readRecord: gets: ex: #{ex}")
        reinit_serial
      end 

      if payload 
        record = UltimeterWeatherMessage.new( payload)

        return nil if !record.is_valid?

        return record
      end

      return nil
    end

  end

end
