#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc at aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/Plugin'
require 'aprs4r/StatusPluginConfiguration'


module APRS4R

  class StatusPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "StatusPlugin")

    attr_reader :aprs4r_version, :aprs4r_uptime, :aprs4r_watchdog, :aprs4r_device
    attr_reader :os_uptime, :os_memory, :os_disk


    def initialize
      logger.info( "iniatialize")

      super

      @start = Time.now

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      return
    end


    def uptime( duration)
      logger.info( "uptime( duration)")

      minute = 60
      hour = 60 * minute 
      day = 24 * hour
      month = 30 * day
      year = 365 * day


      uptime = duration.to_i

      years = (uptime / year).to_i
      uptime %= year

      months = (uptime / month).to_i
      uptime %= month
      
      days = (uptime / day).to_i
      uptime %= day
      
      hours = (uptime / hour).to_i
      uptime %= hour

      minutes = (uptime / minute).to_i
      
      if years > 0
        uptime = sprintf( "%02iy%02im", years, months)
      elsif months > 0
        uptime = sprintf( "%02im%02id", months, days)
      elsif days > 0
        uptime = sprintf( "%02id%02ih", days, hours)
      else
        uptime = sprintf( "%02ih%02im", hours, minutes)
      end

      return uptime
    end


    def create_payload( payload)
      logger.info( "create_payload")

      value = payload.clone

      begin 

        # simple methods, no parameters
        matches = payload.scan( /\#\{ *([a-z0-9]+[a-z0-9\_\-]*) *\}/)
        matches.each{ |match|
          name = match[0].to_s
          
          result = self.send( name.intern)
          logger.debug( "found match: name: #{name}, result: #{result}")
          
          value.gsub!( /\#\{#{name}\}/, result)
          
          logger.debug( "value: #{value}")
        }
        
        # one parameter
        matches = payload.scan( /\#\{ *([a-z0-9]+[a-z0-9\_\-]*) *\( *([a-z0-9]+[a-z0-9\_\-]*) *\) *\}/)
        matches.each{ |match|
          name = match[0].to_s
          parameter = match[1].to_s
          
          logger.debug( "match: #{match.to_s}")
          
          result = self.send( name.intern, parameter)
          logger.debug( "found match: name: #{name}, parameter: #{parameter}, result: #{result}")
          
          value.gsub!( /\#\{ *#{name} *\( *#{parameter} *\) *\}/, result.to_s)
          
          logger.debug( "value: #{value}")
          
        }
      rescue Exception => ex
        logger.warn( "error evaluating status data: #{ex}")
        logger.warn( ex.backtrace.join( "\n"))
        value = nil
      end

      return value
    end


    def aprs4r_version
      version = APRS4RVersion
      version += "-RC-#{APRS4RBuild}" if APRS4RRC == true 

      return version
    end


    def aprs4r_uptime
      return uptime( Time.now - @start)
    end


    def aprs4r_watchdog
      watchdog = "0"
      watchdog = ENV['APRS4R_WATCHDOG_RUN'] if ENV['APRS4R_WATCHDOG_RUN']

      return watchdog
    end


    def aprs4r_device( device)
      send_count = @socket_manager.send_count( device)
      recv_count = @socket_manager.recv_count( device)

      return sprintf( "%i/%i", recv_count, send_count)
    end


    def os_uptime
      value = 0

      begin 
        File.open( "/proc/uptime").each { |line|
          value = line.split( / /)[0].to_i
        }
      rescue Exception
      end

      return uptime( value)
    end


    def os_memory
      memory_total = 0
      memory_free = 0
      begin 
        File.open( "/proc/meminfo").each { |line|
          memory_total = line.chomp!.split( / +/)[1].to_i if line =~ /^MemTotal:/
          memory_free = line.chomp!.split( / +/)[1].to_i if line =~ /^MemFree:/
        }
      rescue Exception
      end

      memory = sprintf( "%3.2f/%3.2fMB", memory_free / 1024.0, memory_total / 1024.0)

      return memory
    end

    
    def os_disk
      disk_total = 0
      disk_free = 0
      
      begin
        output = `df`
        disks = output.split( /\n/) if output

        disks.each{ |disk|
          # search for root partition
          if disk =~ /\/$/
            values = disk.split( / +/)

            disk_total = values[1].to_i
            disk_free = values[3].to_i
          end
        }
      rescue Exception => ex
        logger.error( "Error readig df output: #{ex}")
      end
      
      disk = sprintf( "%3.2f/%3.2fMB", disk_free / 1024.0, disk_total / 1024.0)

      return disk
    end

  end

end
