#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'thread'

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/SocketConfiguration'


module APRS4R

  class Socket < APRS4RBase

    attr_reader :name, :type, :enable, :duplicatePeriod
    
    @logger = APRS4RLogger.get_logger( "Socket")
    

    def initialize
      logger.info( "initialize")

      @thread_mutex = Mutex.new
      @socket_mutex = Mutex.new
      @initialized = false
      
      return
    end


    def setup( configuration = nil)
      logger.info( "setup( configuration)")
      
      @name = nil
      @type = nil
      @enable = false
      @duplicatePeriod = nil
      @txEnable = false

      if configuration
        @name = configuration.name if configuration.name
        @type = configuration.type if configuration.type
        @enable = configuration.enable if configuration.enable
        @duplicatePeriod = configuration.duplicatePeriod if configuration.duplicatePeriod
        @txEnable = configuration.txEnable if configuration.txEnable
      end

      logger.warn( "TX for device #{name} disabled") if @txEnable == false 

      return
    end
    
    
    def delayed_initialize
      logger.info( "delayed_initialize")

      @thread_mutex.synchronize do
        begin
          # delayed initialisation to avoid blocking of main thread
          if !@initialized
            init_socket
            
            @initialized = true 
          end
        rescue Exception => ex
          logger.warn( "delayed_initialize: ex: #{ex}")
        end
      end
      
      return
    end


    def read_message
      logger.info( "read_message")

      message = nil

      begin

        data = read_data
        message = parse_message( data)
        logger.debug( "read_message: message: #{message}")

      rescue SocketError => ex
        logger.warn( "read_message: socket ex: #{ex}")
        raise

      rescue Exception => ex
        logger.warn( "read_message: exception ex: #{ex}")
        logger.warn( ex.backtrace.join( "\n"))
        message = nil
      end
      
      return message
    end
    

    def write_message( message)
      logger.info( "write_message( message)")

      if @txEnable
        begin
          data = serialize_message( message)
          
          logger.debug( "data: #{data}")
          
          write_data( data)

        rescue SocketError => ex
          logger.warn( "write_message: socket: ex: #{ex}")
          raise 

        rescue Exception => ex
          logger.warn( "write_message: ex: #{ex}")
          logger.warn( ex.backtrace.join( "\n"))
        end
      end

      return
    end


    def update_filter( filter_calls)
      logger.info( "update_filter( filter_calls)")

      logger.warn( "no update_filter method implemented")

      return
    end
    
  end

end

