#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/DigipeaterPlugin'
require 'aprs4r/MessagePlugin'

require 'aprs4r/QueryPluginConfiguration'


module APRS4R

  class QueryPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "QueryPlugin")


    def initialize
      logger.info( "initialize")

      super

      @position_message = nil
      @status_message = nil
      @object_message = nil
      @weather_message = nil

      @direct_heard_stations = Hash.new
      @last_direct_heard_expire = Time.now

      @heard_stations = Hash.new
      @last_heard_expire = Time.now
      @last_heard_stations = Array.new

      @message_stations = Array.new
      @last_message_expire = Time.now

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")
      
      super( configuration)
      
      @device = configuration.device

      @call = "" 
      @call = configuration.call if configuration.call
      @call_regexp = // 
      @call_regexp = /#{configuration.call}/
        
      # wildcard call
      if configuration.call =~ /\*$/
        @call = configuration.call.chomp
        @call_regexp = /#{configuration.call.gsub( /\*/, ".*")}/
      end

      @timeout = 900
      begin
        @timeout = configuration.timeout.to_i if configuration.timeout
      rescue Excpetion => ex
        logger.warn( "error converting timeout: #{ex}")
      end

      @stations = 150
      begin
        @stations = configuration.stations.to_i if configuration.stations
      rescue Excpetion => ex
        logger.warn( "error converting stations: #{ex}")
      end

      return
    end
    
    
    def start 
      logger.info( "start")

      if ! @enable
        return
      end

      register_listener( @device)

      return
    end

    
    def register_listener( device)
      logger.info( "register_listener")

      if @enable
        @socket_manager.add_listener( device, self)
        @socket_manager.add_local_listener( device, self)
      end

      return
    end


    def unregister_listener( device)
      logger.info( "unregister_listener")

      @socket_manager.remove_listener( device, self)
      @socket_manager.remove_local_listener( device, self)

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      return if message.nil?

      results = handle_message( name, message)

      if results
        results.each{ |result|
          logger.log( "sending query result on #{@device}, result: #{result}")

          @socket_manager.send_message( name, result)
        }
      end

      return
    end


    def handle_message( name, message)
      logger.info( "handle_message( name, message)")

      return if message.nil?

      return if name != @device

      results = nil

      if message.is_query?
        # handle general queries
        logger.debug( "general query: #{message}")
        
        type = message.query_type

        # general queries
        case type

        when "APRS"
          # general aprs query
          results = handle_APRS( name, message)
        when "DIGI"
          # general digi query
          results = handle_DIGI( name, message)
        when "IGATE"
          # general igate query
          results = handle_IGATE( name, message)
        when "WX"
          # general wx query
          results = handle_WX( name, message)
        end
        
      elsif message.is_message? && message.message_recipient =~ @call_regexp
        # handle directed queries
        logger.debug( "directed query: #{message}")

        type = message.query_type

        case type

        when "APRSD"
          # direct heard
          results = handle_APRSD( name, message)

        when "APRSH"
          # heard
          results = handle_APRSH( name, message) 
          
        when "APRSM"
          # message query
          results = handle_APRSM( name, message)

        when "APRSO"
          # object query
          results = handle_APRSO( name, message)

        when "APRSP"
          # position query
          results = handle_APRSP( name, message)

        when "APRSS"
          # status query
          results = handle_APRSS( name, message)
          
        when "APRST" 
          # trace/ping query
          results = handle_APRST( name, message)

        when "PING?"
          # trace/ping query
          results = handle_APRST( name, message)
          
        end

      else
        # handle all other aprs messages
        logger.debug( "other aprs message: #{message}")
        
        # add direct heard 
        add_direct_heard( message) if message.is_direct?
        
        # add heard 
        add_heard( message)

        # add message
        add_message( message) if message.is_message?

      end
      
      return results
    end


    def recv_local_message( name, message)
      logger.info( "recv_local_message( name, message)")

      # fill messages
      if message.source.to_s =~ @call_regexp

        if message.has_position?
          @position_message = message.clone 
          logger.debug( "position: #{@position_message}")
        end

        if message.has_status?
          @status_message = message.clone
          logger.debug( "status: #{@status_message}")
        end

        if message.is_object?
          @object_message = message.clone
          logger.debug( "object: #{@object_message}")
        end

        if message.has_weather?
          @wx_message = message.clone 
          logger.debug( "wx: #{@wx_message}")
        end

      end

      return
    end


    def handle_APRS( name, message)
      logger.info( "handle_APRS( name, message)")

      results = Array.new

      # send position report
      results.concat( handle_APRSP( name, message))

      # send status report
      results.concat( handle_APRSS( name, message))

      return results
    end


    def handle_DIGI( name, message)
      logger.info( "handle_DIGI( name, message)")

      result = nil
      results = Array.new

      total_count = 0
      direct_count = 0

      plugin = @plugin_manager.get_plugin( DigipeaterPlugin)

      if plugin 
        total_count = plugin.total_count
        direct_count = plugin.direct_count
      end
      
      result = APRSMessage.new( @call, APRS4RCall)

      # TODO path handling
      result.path = ["WIDE3-3"]
      
      payload = sprintf( "<DIGI,TOTAL_CNT=%i,DIRECT_CNT=%i", total_count, direct_count)
      result.payload = payload

      results << result

      return results
    end

    
    def handle_IGATE( name, message)
      logger.info( "handle_IGATE( name, message)")

      result = nil
      results = Array.new

      message_count = 0
      local_count = 0

      plugin = @plugin_manager.get_plugin( MessagePlugin)

      if plugin
        message_count = plugin.message_count
        local_count = plugin.local_count
      end

      result = APRSMessage.new( @call, APRS4RCall)

      # TODO path handling
      result.path = ["WIDE3-3"]
      
      payload = sprintf( "<IGATE,MSG_CNT=%i,LOC_CNT=%i", message_count, local_count)
      result.payload = payload

      results << result
      
      return results
    end


    def handle_WX( name, message)
      logger.info( "handle_WX( name, message)")
      
      results = Array.new

      # send wx report
      if @wx_message
        @wx_message.path = []
        results << @wx_message
      end

      # send position report
      results.concat( handle_APRSP( name, message))

      return results
    end

    
    def handle_APRSD( name, message)
      logger.info( "handle_APRSD( name, message)")
      
      expire_direct_heard
      @last_direct_expire = Time.now

      results = Array.new

      stations = String.new
      @direct_heard_stations.each_key{ |station|
        stations += " #{station}"
      }

      response = APRSMessage.new( @call, APRS4RCall)
      response_text = "Directs=#{stations}"
      response.payload = ":" + message.source[0..8].ljust( 9) + ":" + response_text

      results << response

      return results
    end


    def handle_APRSH( name, message)
      logger.info( "handle_APRSH( name, message)")

      results = Array.new

      call = message.query_payload
      timestamps = @heard_stations[call]

      if !timestamps.nil? && !timestamps.empty?
        response = APRSMessage.new( @call, APRS4RCall)
        response_text = "#{call} HEARD:"

        now = Time.now
        hours = Array.new( 8, 0)

        timestamps.each{ |timestamp|
          index = (now - timestamp) % (60 * 60)

          hours[index] += 1 if index >= 0 && index < 8
        }

        hours.each{ |hour|
          if hour > 0 
            response_text += " #{hour}"
          else
            response_text += " ."
          end
        }

        response.payload = ":" + message.source[0..8].ljust( 9) + ":" + response_text

        results << response
      end

      return results
    end


    def handle_APRSM( name, message)
      logger.info( "handle_APRSM( name, message)")

      results = Array.new

      recipient = message.source

      @message_stations.each{ |entry|
        entry_message = entry[0]
        if recipient == entry_message.message_recipient
          results << entry_message
        end
      }
        
      return results
    end


    def handle_APRSO( name, message)
      logger.info( "handle_APRSO( name, message)")

      results = Array.new

      if @object_message
        # TODO path handling
        @object_message.path = []
        results << @object_message
      end

      return results
    end


    def handle_APRSP( name, message)
      logger.info( "handle_APRSP( name, message)")

      results = Array.new
    
      if @position_message
        # TODO path handling
        @position_message.path = []
        results << @position_message
      end

      return results
    end
      

    def handle_APRSS( name, message)
      logger.info( "handle_ARPSS( name, message)")

      results = Array.new

      if @status_message
        # TODO path handling
        @status_message.path = []
        results << @status_message
      end

      return results
    end


    def handle_APRST( name, message)
      logger.info( "handle_APRST( name, message)")

      results = Array.new

      result = APRSMessage.new( @call, APRS4RCall)

      # TODO path handling
      result.path = ["WIDE3-3"]

      payload = sprintf( ":%9.9s:%s>%s", message.source.ljust( 9), message.destination, message.path.join( ','))

      result.payload = payload

      results << result
      
      return results
    end


    def add_direct_heard( message)
      logger.info( "add_direct_heard( message)")

      @direct_heard_stations[message.source] = Time.now

      if (@direct_heard_stations.size > (@stations + 10)) ||
         (@last_direct_heard_expire + @timeout < Time.now)
        expire_direct_heard
        @last_direct_heard_expire = Time.now
      end
      
      return
    end


    def expire_direct_heard
      logger.info( "expire_direct_heard")

      now = Time.now

      # remove expired stations
      @direct_heard_stations.each{ |key, timestamp|
        if now - timestamp > @timeout
          delete_direct_heard( key)
        end
      }

      if @direct_heard_stations.size > @stations
        stations = @direct_heard_stations.sort{ |a, b| a[1] <=> b[1] }

        for i in @stations...stations.length
          delete_direct_heard( stations[i][0])
        end
      end

      return
    end

    
    def delete_direct_heard( call)
      logger.info( "delete_direct_heard( call)")
      
      @direct_heard_stations.delete( call)

      return
    end


    def add_heard( message)
      logger.info( "add_heard( message)")

      return if message.nil?

      now = Time.now

      timestamps = @heard_stations[message.source]

      timestamps = Array.new if timestamps.nil?
      timestamps << now

      @heard_stations[message.source] = timestamps

      @last_heard_stations.delete( message.source)
      @last_heard_stations.unshift( message.source)

      if (@heard_stations.size > (@stations + 10)) ||
         (@last_heard_expire + @timeout < now)
        expire_heard
        @last_heard_expire = now
      end

      return
    end

    
    def expire_heard
      logger.info( "expire_heard")

      while @last_heard_stations.size > @stations
        call = @last_heard_stations.pop
        
        delete_heard( call)
      end

      now = Time.now
      @heard_stations.each_value{ |timestamps|
        while timestamps.first && (now - timestamps.first > (8 * 60 * 60))
          timestamps.pop
        end
      }

      return
    end


    def delete_heard( call)
      logger.info( "delete_heard( call)")
      
      @heard_stations.delete( call)
      @last_heard_stations.delete( call)

      return
    end


    def add_message( message)
      logger.info( "add_message( message)")

      return if message.nil?

      if message.is_message?
        logger.debug( "incoming message: #{message}")
        @message_stations << [message, Time.now]
      end

      if (@message_stations.size > (@stations + 10)) ||
         (@last_message_expire + @timeout < Time.now) 
        expire_message
        @last_message_expire = Time.now
      end

      return
    end


    def expire_message
      logger.info( "expire_message")

      while @message_stations.size > @stations
        @message_stations.pop
      end

      now = Time.now
      while (@message_stations.size > 0) && (now - @message_stations.last[1] > @timeout)
        @message_stations.pop
      end

      return
    end

  end

end
