#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'


module APRS4R

  class PluginManager < APRS4RBase

    @logger = APRS4RLogger.get_logger( "PluginManager")

    @@instance = nil


    def initialize
      logger.info( "initialize")

      @configuration = nil
      @plugins = Hash.new

      return
    end

    
    def PluginManager.get_instance
      logger.info( "get_instance")

      @@instance = PluginManager.new if @@instance.nil?

      return @@instance
    end


    def start( configuration)
      logger.info( "start( configuration)")

      return if configuration.nil?

      @configuration = configuration.plugins

      return if @configuration.nil?

      @configuration.each{ |key, configuration| 
        begin
          name = configuration.name
          type = configuration.type
          enable = configuration.enable
      
          if enable 
            logger.debug( "loading plugin: #{type}")
            plugin = load_plugin( type)

            logger.debug( "setting up plugin: #{plugin}")
            setup_plugin( plugin, configuration)

            logger.debug( "starting plugin: #{plugin}")
            start_plugin( plugin)

            add_plugin( plugin, configuration)
          end

        rescue Exception => ex
          logger.error( "error loading plugin: #{name}, ex: #{ex}")
          logger.error( ex.backtrace.join( "\n"))
        end

      }

      logger.debug( "all plugins loaded")

      return
    end


    def reload
      logger.info( "reload")


      @plugins.each{ |plugin, configuration|
        begin
          
          logger.debug( "checking plugin: #{plugin}")
          
          name = configuration.name
          type = configuration.type
          enable = configuration.enable
          remote = configuration.remote
          changed = configuration.changed
      
          if remote && changed
            logger.debug( "configuration changed for plugin: #{plugin}")

            logger.debug( "stopping plugin: #{plugin}")
            stop_plugin( plugin)

            logger.debug( "setting up plugin: #{plugin}")
            setup_plugin( plugin, configuration)
            
            logger.debug( "starting plugin: #{plugin}")
            start_plugin( plugin)
            
            add_plugin( plugin, configuration)
          end

        rescue Exception => ex
          logger.error( "error reloading plugin: #{name}, ex: #{ex}")
          logger.error( ex.backtrace.join( "\n"))
        end

      }

      return
    end


    def load_plugin( type)
      logger.info( "load_plugin( type)")

      return nil if type.nil?
      logger.debug( "type: #{type}")
        
      plugin = nil
      plugin_class = "aprs4r" + "/" + type
        
      logger.debug( "plugin_class: #{plugin_class}")
      require plugin_class

      name = type.split( /\//).last
      
      plugin = eval( "#{name}.new")
      logger.debug( "plugin: #{plugin}")
      
      return plugin
    end
       

    def setup_plugin( plugin, configuration)
      logger.info( "setup_plugin( plugin, configuration)")

      return if plugin.nil? || configuration.nil?

      plugin.setup( configuration) 

      return
    end


    def start_plugin( plugin)
      logger.info( "start_plugin( plugin)")

      return if plugin.nil?
      
      plugin.start

      return
    end

    
    def stop_plugin( plugin)
      logger.info( "stop_plugin( plugin)")

      return if plugin.nil?
      
      plugin.stop

      return
    end

    
    def add_plugin( plugin, configuration)
      logger.info( "add_plugin")
      
      @plugins[plugin] = configuration
      
      return
    end


    def delete_plugin( plugin)
      logger.info( "delete_plugin( plugin)")
      
      @plugins.delete( plugin)

      return
    end
    
    
    def get_plugin( type)
      logger.info( "get_plugin")
      
      @plugins.each_key{ |plugin|
        return plugin if plugin.is_a?( type)
      }
      
      return nil
    end


  end


end
