#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'socket'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/ISServerPluginConfiguration'
require 'aprs4r/ISServerSocket'

require 'aprs4r/Plugin'


module APRS4R

  class ISServerPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "ISServerPlugin")


    def initialize
      logger.info( "initialize")

      super

      @thread_run = false
      @thread = nil

      return
    end

    
    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @port = configuration.port

      @path = []
      @path = configuration.path if configuration.path

      @inDevices = Array.new
      @inDevices = configuration.inDevices if configuration.inDevices
      
      @outDevices = Array.new
      @outDevices = configuration.outDevices if configuration.outDevices

      @serverSocket = ISServerSocket.new( @port)
      @clientSockets = Array.new

      return
    end

    
    def start
      logger.info( "start")

      return if !@enable

      @thread_run = true
      @thread = Thread.new{ runServerThread }

      @inDevices.each{ |device|
        register_listener( device)
      }

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable
      
      # TODO stop server thread and client threads

      @inDevices.each{ |device|
        unregister_listener( device)
      }

      return
    end


    def register_listener( device)
      logger.info( "register_listener")

      if @enable
        @socket_manager.add_listener( device, self)
        @socket_manager.add_local_listener( device, self)
      end

      return
    end


    def unregister_listener( device)
      logger.info( "unregister_listener")

      if @enable
        @socket_manager.remove_listener( device, self)
        @socket_manager.remove_local_listener( device, self)
      end

      return
    end


    def runServerThread
      logger.info( "run")

      while true 
        clientSocket = @serverSocket.accept

        logger.log( "clientSocket: #{clientSocket}")

        if clientSocket 
          Thread.new { runClientThread( clientSocket) } 
          @clientSockets << clientSocket
        end

      end
    end


    def runClientThread( clientSocket)
      logger.info( "runClientThead( clientSocket)")

      run = true 

      while run
        begin
          message = clientSocket.read_message
          logger.log( "recv local is message: #{message}")

          if !message.nil?
            path = message.path
            
            if path 
              if path.length == 1 && path[0] =~ /TCPIP/
                message.path = @path
              end
            end
            
            @outDevices.each{ |device|
              @socket_manager.send_message( device, message)
            }
          end
          
        rescue Exception => ex
          logger.warn( "runClientThread::ex: " + ex)
          @clientSockets.delete( clientSocket)
          run = false
        end
      end

      return
    end


    def recv_local_message( name, message)
      logger.info( "recv_local_message( name, message)")

      recv_message( name, message)
    end

    
    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      if message.nil? 
        return
      end

      @clientSockets.each{ |clientSocket|
        
        begin
          logger.log( "send local is message: #{message}")
          clientSocket.write_message( message)
        rescue Exception => ex
          logger.warn( "write_message: ex: #{ex}")
          @clientSockets.delete( clientSocket)
        end

      }
      
      return
    end

  end

end
