#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/DigipeaterPluginConfiguration'
require 'aprs4r/Plugin'


module APRS4R

  class DigipeaterPlugin < Plugin

    @@RFONLY = "RFONLY"

    @logger = APRS4RLogger.get_logger( "DigipeaterPlugin")


    def initialize
      logger.info( "initialize")

      super

      @total_count = 0
      @direct_count = 0

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @device = configuration.device
      @call = configuration.call

      @mode = "fillin" 
      @mode = "wide" if configuration.mode == "wide"

      @aliases = Array.new
      if configuration.aliases
        configuration.aliases.each{ |entry|
          call = APRSCall.parse( entry)
          @aliases << call
        }
      end

      @state = nil
      @state = configuration.state if configuration.state

      @rfonly = false
      @rfonly = configuration.rfonly if configuration.rfonly

      @translatePath = true
      @translatePath = configuration.translatePath if !configuration.translatePath.nil?

      @reducePath = false
      @reducePath = configuration.reducePath if configuration.reducePath
      
      begin
        @permittHops = 7
        @permittedHops = configuration.permittedHops.to_i if configuration.permittedHops
      rescue Exception => ex
        logger.warn( "error converting parameter #{ex}")
      end
      
      if @call.to_s =~ /^MYCALL/i
        raise "DigipeaterPlugin: plugin: #{@name}, invalid call: #{@call}, please use valid call"
      end

      return
    end


    def start
      logger.info( "start")

      return if !@enable
      
      register_listener
      
      return
    end
    
    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      return if !@enable

      @socket_manager.add_listener( @device, self)
      
      return
    end
    
    
    def unregister_listener
      logger.info( "unregister_listener")

      return if !@enable

      @socket_manager.remove_listener( @device, self)
      
      return
    end
    
    
    def recv_message( name, message)
      logger.info( "recv_message( name, message)")
      
      return if message.nil?

      result = do_digipeating( name, message)

      @total_count += 1 if result
      @direct_count += 1 if message.is_local?

      result = do_post_check( result)
      
      if result
        logger.log( "digipeating on #{@device}, message: #{result}")
        @socket_manager.send_message( @device, result)
      end
      
      return
    end
    
    
    def do_digipeating( name, message)
      logger.info( "do_digipeating( name, message)")

      return nil if message.nil?
      
      return nil if message.path.nil? || message.path.empty?
      
      # translate path
      if @translatePath
        message = translate_path( message)
        logger.debug( "do_digipeating: path translate: #{message}")
      end

      return nil if message.nil? || message.path.nil?
      
      # optimize path 
      message = optimize_path( message)
      logger.debug( "do_digipeating: path optimize: #{message}")

      return nil if message.nil? || message.path.nil?
      
      path = message.path
      
      # reduce path (if enabled)
      if @reducePath
        rfonly = false
        path.each{ |entry|
          rfonly = true if entry == @@RFONLY
        }
        
        if (@rfonly == false) || ((@rfonly == true) && (rfonly == false))
          message = reduce_path( message)
          logger.debug( "do_digipeating: path reduce: #{message}")
        end
      end

      return nil if message.nil? || message.path.nil? || message.path.empty?
      
      path = message.path
      
      # search for last repeated flag
      repeated_index = 0
      for i in 0...path.length
        entry = APRSCall.parse( path[i])
        
        repeated_index = i+1 if entry.repeated
      end
      
      repeat = false
      wide_keyword = /WIDE1/
      wide_keyword = /(WIDE|#{@state})1/ if @state
      if @mode == "wide"
        wide_keyword = /WIDE[1-7]/
        wide_keyword = /(WIDE|#{@state})[1-7]/ if @state
      end
      
      # search for keyword
      for i in repeated_index...path.length
        entry = APRSCall.parse( path[i])
        
        # check for wide keyword
        if entry.call =~ wide_keyword && entry.ssid > 0 
          repeat = true
          
          entry.decrease_ttl
          digi = APRSCall.parse( @call)

          if entry.ssid == 0 
            path.delete_at( i)
          else
            path[i] = entry.to_s
          end
          
          digi.repeated = true 
          path.insert( i, digi.to_s)
          
          return message
        end
        
        # check for aliases (and replace and disable digipeater call)
        if @aliases.include?( entry)
          call = APRSCall.parse( @call)
          call.repeated = true 
          path[i] = call.to_s
          
          return message
        end
        
      end
      
      return nil
    end

    
    def translate_path( message)
      logger.info( "translate_path( message)")
      
      return nil if message.nil? || message.path.nil?

      path = message.path

      # search for last repeated flag
      repeated_index = 0
      for i in 0...path.length
        entry = APRSCall.parse( path[i])

        repeated_index = i+1 if entry.repeated
      end

      behind_relay = false
      behind_wide1 = false

      for i in repeated_index...path.length
        entry = path[i]
        call = nil

        if entry == "RELAY"
          if behind_relay
            call = APRSCall.parse( "WIDE2-1")
          else
            call = APRSCall.parse( "WIDE1-1")
            behind_relay = true
            behind_wide1 = true
          end
        elsif entry == "WIDE" || entry == "TRACE"
          call = APRSCall.parse( "WIDE2-1")
        elsif entry == "WIDE1-1"
          if behind_wide1
            call = APRSCall.parse( "WIDE2-1")
          else
            bedind_wide1 = true
          end
        elsif entry =~ /TRACE[0-9]/
          call = APRSCall.parse( "WIDE" + entry[5...8])
        end

        path[i] = call.to_s if call
      end

      return message
    end
    

    def optimize_path( message)
      logger.debug( "optimizePath( message)")

      return nil if message.nil? || message.path.nil?

      path = message.path
      path_new = Array.new

      # search for last repeated flag
      repeated_index = 0
      for i in 0...path.length
        entry = APRSCall.parse( path[i])

        repeated_index = i+1 if entry.repeated
      end

      logger.debug( "optimize_path: repeated_index: #{repeated_index}")

      for i in 0...repeated_index 
        logger.debug( "optimize_path: adding entry: #{path[i]}")
        path_new << path[i]
      end

      last_wide_ssid = 0
      for i in repeated_index...path.length
        entry = path[i]
        logger.debug( "entry: #{entry}")
        call = APRSCall.parse( entry)
        current_ssid = call.ssid

        if call.call =~ /WIDE[2-9]/ && call.ssid > 0 
          if last_wide_ssid + current_ssid <= 7
            last_wide_ssid += current_ssid
          else 
            wide_new = "WIDE" + last_wide_ssid.to_s + "-" + last_wide_ssid.to_s
            wide_new = "WIDE2-1" if wide_new == "WIDE1-1"
            path_new << wide_new
            last_wide_ssid = current_ssid
          end
        else     
          logger.debug( "optimize_path: adding entry: #{call}")
          path_new << entry
        end

      end

      if last_wide_ssid > 0 
        wide_new = "WIDE" + last_wide_ssid.to_s + "-" + last_wide_ssid.to_s
        wide_new = "WIDE2-1" if wide_new == "WIDE1-1"
        path_new << wide_new
      end

      message.path = path_new

      return message
    end


    def reduce_path( message) 
      logger.info( "reduce_path( message)")

      return nil if message.nil? || message.path.nil?

      path = message.path
      path_new = Array.new
      
      if @permittedHops.nil?
        logger.warn( "permittedHops not set")
        return nil
      end

      ssid_sum = 0

      # search for last repeated flag
      repeated_index = 0
      for i in 0...path.length
        entry = APRSCall.parse( path[i])

        repeated_index = i+1 if entry.repeated
        
        if entry.call =~ /WIDE[2-9]/ || entry.call =~ /TRACE[2-9]/
          ssid_sum += entry.ssid
        end
      end

      logger.debug( "optimize_path: repeated_index: #{repeated_index}")

      for i in 0...repeated_index 
        logger.debug( "optimize_path: adding entry: #{path[i]}")
        path_new << path[i]
      end

      logger.debug( "ssid_sum: #{ssid_sum}")
      logger.debug( "permittedHops: #{@permittedHops}")

      for i in repeated_index...path.length
        entry = path[i]
        call = APRSCall.parse( entry)

        if entry =~ /WIDE[2-9]/ 
          ssid = call.ssid

          if ssid_sum > @permittedHops

            logger.debug( "in if: ssid_sum: #{ssid_sum}, permittedHops: #{@permittedHops}")
            
            decrement = [ ssid_sum - @permittedHops, ssid].min

            call.ssid -= decrement
            ssid_sum -= decrement

            if call.ssid != 0 
              path_new << call.call[0...call.call.length-1] + [2, call.ssid].max.to_s + "-" + call.ssid.to_s
            end
          else
            path_new << entry
          end
          
        else
          path_new << entry
        end
      end

      message.path = path_new

      return message
    end

    
    def do_post_check( message)
      logger.info( "do_post_check")

      return nil if message.nil? || message.path.nil?

      path = Array.new

      message.path.each{ |entry|
        
        if entry == "RELAY"
          # do not use old RELAY
          logger.error( "do_post_check: RELAY found")
        elsif entry == "WIDE"
          # do not use old WIDE
          logger.error( "do_post_check: WIDE found")
        elsif entry == "TRACE"
          # do not use old TRACE
          logger.error( "do_post_check: TRACE found")
        elsif entry == "WIDE1-1"
          # after first digipeating, no WIDE1-1 allowed
          logger.error( "do_post_check: WIDE1-1 found")
        elsif entry =~ /TRACE[0-9]/
          # do not use old TRACEn-n
          logger.error( "do_post_check: TRACEn-n found")
        else
          path << entry
        end
        
      }

      message.path = path
      
      return message
    end


    def total_count
      return 0
    end

    
    def direct_count
      return 0 
    end

  end

end
