#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2010 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'

require 'aprs4r/extensions/Extension'
require 'aprs4r/extensions/ReducePathExtensionConfiguration'


module APRS4R

  class ReducePathExtension < Extension

    @logger = APRS4RLogger.get_logger( "ReducePathExtension")


    def initialize
      logger.info( "initialize")

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      begin
        @permittedHops = 7
        @permittedHops = configuration.permittedHops.to_i if configuration.permittedHops
      rescue Exception => ex
        logger.warn( "error converting parameter: #{ex}")
      end
      
      return
    end


    def handle_message( name, message, repeat)
      logger.info( "handle_message( name, message, repeat)")

      return [@@RESULT_ACCEPT, nil] if message.nil? || message.path.nil?

      path = message.path
      path_new = Array.new
      
      if @permittedHops.nil?
        logger.warn( "permittedHops not set")
        return [@@RESULT_ACCEPT, message]
      end

      ssid_sum = 0

      # search for last repeated flag
      repeated_index = 0
      for i in 0...path.length
        entry = APRSCall.parse( path[i])

        repeated_index = i+1 if entry.repeated
        
        if entry.call =~ /WIDE[2-9]/ || entry.call =~ /TRACE[2-9]/
          ssid_sum += entry.ssid
        end
      end

      logger.debug( "optimize_path: repeated_index: #{repeated_index}")

      for i in 0...repeated_index 
        logger.debug( "optimize_path: adding entry: #{path[i]}")
        path_new << path[i]
      end

      logger.debug( "ssid_sum: #{ssid_sum}")
      logger.debug( "permittedHops: #{@permittedHops}")

      for i in repeated_index...path.length
        entry = path[i]
        call = APRSCall.parse( entry)

        if entry =~ /WIDE[2-9]/ 
          ssid = call.ssid

          if ssid_sum > @permittedHops

            logger.debug( "in if: ssid_sum: #{ssid_sum}, permittedHops: #{@permittedHops}")
            
            decrement = [ ssid_sum - @permittedHops, ssid].min

            call.ssid -= decrement
            ssid_sum -= decrement

            if call.ssid != 0 
              path_new << call.call[0...call.call.length-1] + [2, call.ssid].max.to_s + "-" + call.ssid.to_s
            end
          else
            path_new << entry
          end
          
        else
          path_new << entry
        end
      end

      message.path = path_new

      return [@@RESULT_ACCEPT, message]
    end

  end

end
