#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'thread'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/GatewayPlugin'
require 'aprs4r/BuddyGatewayPluginConfiguration'


module APRS4R

  class BuddyGatewayPlugin < GatewayPlugin

    @logger = APRS4RLogger.get_logger( "BuddyGatewayPlugin")


    def initialize
      logger.info( "initialize")

      super

      @mutex = Mutex.new
      @thread = nil
      @thread_run = false
      @thread_mutex = Mutex.new

      @timeout = 12 * 60 * 60 # 12 hours
      @update_timeout = 5 * 60 # 5 minutes
      @heard_timeout = 15 * 60 # 15 minutes

      @heard_calls = Hash.new

      @filter_calls = Hash.new
      @filter_regexps = Hash.new

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @buddies = []
      @buddies = configuration.buddies if configuration.buddies

      begin
        @timeout = configuration.timeout.to_i if configuration.timeout
      rescue Exception => ex
        logger.error( "Error reading timeout variable: #{configuration.timeout}")
      end

      # add buddies to filter
      @buddies.each{ |buddy|
        add_filter( buddy, nil)
      }

      update_filter

      return
    end


    def start
      logger.info( "start")

      return if !@enable

      register_listener

      @thread_run = true
      @thread = Thread.new { run_update_thread } 

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      @thread_run = false
      
      @thread_mutex.synchronize do 
        @thread.run

        while @thread.status != false && @thread.status != nil
          sleep 1
          @thread.run 
        end
      end
        
      return
    end


    def register_listener
      logger.info( "register_listener")

      return if !@enable

      @socket_manager.add_listener( @inDevice, self)
      @socket_manager.add_listener( @outDevice, self)

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      return if !@enable

      @socket_manager.remove_listener( @inDevice, self)
      @socket_manager.remove_listener( @outDevice, self)

      return
    end


    def run_update_thread
      logger.info( "run_update_thread")

      while @thread_run

        @thread_mutex.synchronize do 
          update_filter
        end

        sleep @update_timeout
      end

      return
    end


    def recv_message( name, message)
      logger.info( "recv_messag( name, message)")

      handle_ack( name, message)

      result = handle_message( name, message)
      
      if result 
        @socket_manager.send_message( @outDevice, result)
      end

    end


    def handle_ack( name, message)
      logger.info( "handle_ack( message)")

      # send ack if necessary 
      if name == @outDevice && message.is_message? && message.message_recipient == @call.to_s
        if message.message_ack
          ack = APRSMessage.new( @call, APRS4RCall)
          ack.payload = ":" + message.source[0..8].ljust( 9) + ":" + "ack" + message.message_ack
          
          @socket_manager.send_message( @outDevice, ack)
        end
      end
      
    end
      

    def handle_message( name, message)
      logger.info( "handle_message( name, message)")
      
      return if message.nil?

      if name == @inDevice

        expire_filters if in_filter?( message.source)

        if in_filter?( message.source)
          # station is in filter 

          # but station was heard locally
          return if @heard_calls.has_key?( message.source)

          # replace path and sending beacon in outDevice
          message.path = @path

          logger.log( "gating buddy from #{name} to #{@outDevice}, message: #{message}")
          return message
        end
        
      elsif name == @outDevice

        if in_filter?( message.source)

          # store station call of locally heard station
          @heard_calls[message.source] = Time.now
        elsif message.is_message? && message.message_recipient == @call.to_s

          payload = message.message_payload

          return if payload.nil?
            
          if payload =~ /^\?APRSB/i
            # query buddy call
            values = payload.split( / /)

            if values.length >= 2
              buddy_call = values[1]
              
              response = APRSMessage.new( @call, APRS4RCall)
              response_text = nil
              
              if in_filter?( buddy_call)
                # send positive response
                response_text = "BUDDY #{buddy_call} is already in buddy list."
              else
                # send negative response
                response_text = "BUDDY #{buddy_call} is not in buddy list"
              end
              
              response.payload = ":" + message.source[0..8].ljust( 9) + ":" + response_text
              
              return response
            end

          elsif payload =~ /^\!APRSB/i
            # set buddy call
            values = payload.split( / /)
            
            if values.length >= 2
              buddy_call = values[1]
              
              remove_filter( buddy_call)
              add_filter( buddy_call)
              update_filter
              
              response_message = "BUDDY #{buddy_call} added to buddy list (usally 12h)"
              
              response = APRSMessage.new( @call, APRS4RCall)
              response.payload = ":" + message.source[0..8].ljust( 9) + ":" + response_message
              
              return response
            end
          end
        end
      end
        
      return nil
    end
  
  
    def expire_filters
      logger.info( "expire_filters")

      now = Time.now

      @mutex.synchronize do 
        # clean up local station list 
        @heard_calls.each{ |key, timestamp|
          if now - timestamp > @heard_timeout
            @heard_calls.delete( key)
          end
        }

        # clean up filter station list 
        @filter_calls.each{ |key, timestamp|
          if timestamp && (now - timestamp > @timeout)
            remove_filter( key)
          end
        }
      end
      
      return
    end


    def add_filter( buddy_call, time = Time.now)
      logger.info( "add_filter( buddy_call)")

      return if buddy_call.nil? || buddy_call.empty?
      
      buddy_regexp = /#{buddy_call.gsub( /\*/, ".*")}/

      @filter_calls.store( buddy_call, time)
      @filter_regexps.store( buddy_call, buddy_regexp)

      return
    end
    
    
    def remove_filter( buddy_call)
      logger.info( "remove_filter( buddy_call)")

      return if buddy_call.nil? || buddy_call.empty?

      @filter_calls.delete( buddy_call)
      @filter_regexps.delete( buddy_call)

      return
    end
    
    
    def in_filter?( buddy)
      logger.info( "in_filter?( buddy)")

      return false if buddy.nil? || buddy.empty?
      
      @filter_regexps.each{ |key, filter|
        return true if buddy =~ filter
      }
      
      return false
    end


    def update_filter
      logger.info( "update_filter")

      @mutex.synchronize do
        logger.debug( "updating #{@inDevice} filter: #{@filter_calls.keys}")
        @socket_manager.update_filter( @inDevice, @filter_calls.keys)
      end

      return
    end

  end

end
