#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2010 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RTest'

require 'aprs4r/APRSMessage'

require 'aprs4r/extensions/Extension'
require 'aprs4r/extensions/DigipeaterExtension'
require 'aprs4r/extensions/DigipeaterExtensionConfiguration'


module APRS4R 

  class FillinDigipeaterExtensionTest < APRS4RTest

    @logger = APRS4RLogger.get_logger( "FillinDigipeaterExtensionTest")


    def setup
      logger.info( "setup")

      configuration = DigipeaterExtensionConfiguration.new
      configuration.name = "fillin"
      configuration.type = "DigipeaterExtension"
      configuration.enable = true

      configuration.call = "DIGI"
      configuration.mode = "fillin"
      configuration.aliases = [ "DIGI1", "DIGI2" ]
      configuration.state = nil
      configuration.rfonly = false

      @extension = DigipeaterExtension.new
      @extension.setup( configuration)

      @device = "test0"
      @repeat = false

      return
    end


    def test_nil
      logger.info( "test_nil")
      
      message1 = nil

      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_ACCEPT)
      assert_nil( result1)
    end


    def test_simple_wide2
      logger.info( "test_simple_wide2")

      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, [ "DL1LJ*", "WIDE2-1"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_DROP)
      assert_nil( result1)

      return
    end


    def test_simple_wide3
      logger.info( "test_simple_wide3")

      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, [ "DL1LJ*", "WIDE3-2"], "")
    
      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_DROP)
      assert_nil( result1)

      return
    end


    def test_used_wide1
      logger.info( "test_used_wide1")

      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, ["WIDE1", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_DROP)
      assert_nil( result1)

      return
    end
    

    def test_wide1
      logger.info( "test_wide1")
      
      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, ["WIDE1-1", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI*", "WIDE2-2"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)

      return
    end


    def test_alias
      logger.info( "test_alias")
      
      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, ["DIGI2", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI*", "WIDE2-2"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)

      return
    end


    def test_unusable_alias
      logger.info( "test_unusable_alias")
      
      message1 = APRSMessage.new( "DO5MC", APRS4R::APRS4RCall, ["DIGI1", "DIGI5*", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_DROP)
      assert_nil( result1)

      return
    end


  end

end
  
