#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'thread'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/XMLDataPluginConfiguration'


module APRS4R

  class XMLDataPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "XMLDataPlugin")


    def initialize
      logger.info( "initialize")

      super

      @distances = Array.new

      @timestamps = Hash.new
      @messages = Hash.new

      @symbols = Hash.new
      @positions = Hash.new
      @comments = Hash.new
      @states = Hash.new
      @weathers = Hash.new
      @courses = Hash.new
      @phgs = Hash.new

      @thread = nil
      @thread_run = false
      @thread_mutex = Mutex.new
      
      return
    end
    

    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @file = configuration.file
      @device = configuration.device

      @call = configuration.call
      @latitude = configuration.latitude
      @longitude = configuration.longitude

      # use default value if timeout is missing
      @timeout = 1800
      @timeout = configuration.timeout if configuration.timeout

      # use default value if period is missing
      @period = 30
      @period = configuration.period if configuration.period

      # use default value (all accepting regexp) if filter is missing
      @filter = //
      @filter = Regexp.new( configuration.filter) if configuration.filter

      # limit number of stations to export 
      @stations = 150
      @stations = configuration.stations.to_i if configuration.stations

      return
    end

    
    def start
      logger.info( "start")

      return if !@enable

      register_listener

      @thread_run = true
      @thread = Thread.new{ run } 

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      if @enable
        @socket_manager.add_listener( @device, self)
        @socket_manager.add_local_listener( @device, self)
      end

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      if @enable
        @socket_manager.remove_listener( @device, self)
        @socket_manager.remove_local_listener( @device, self)
      end

      return
    end


    def recv_local_message( name, message) 
      logger.info( "recv_local_message( name, message)")

      recv_message( name, message)

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      if message.nil? 
        return
      end

      @thread_mutex.synchronize do

        source = message.source.to_s

        if message.is_object?
          source = message.object_address
          logger.debug( "is_object? == true, source: #{source}")
        end


        if source =~ @filter

          now = Time.now

          if @stations > 0 
            distance = nil
            distance = message.distance( @latitude, @longitude) if message.has_position?

            insert_station( source, distance)

            if @distances.length >= (@stations + 10)

              while @distances.length > @stations
                last_source = last_distance = nil
                last_source, last_distance = @distances.last
                
                delete_station( last_source, last_distance)
              end

            end

          end


          @timestamps[source] = now
          @messages[source] = message

          if message.has_symbol? 
            logger.debug( "message: symbol: #{message.symbol_code} name: #{message.symbol_name}")
            @symbols[source] = message
          end
          
          if message.has_position?
            @positions[source] = message
            # logger.warn( "message: lat: #{message.latitude} long: #{message.longitude}")
          end
          
          if message.has_comment?
            @comments[source] = message
            # logger.warn( "message: comment: #{message.comment}")
          end
          
          if message.has_status?
            @states[source] = message
            # logger.warn( "message: status: #{message.status}")
          end
          
          if message.has_weather?
            @weathers[source] = message
            # logger.warn( "message: weather: wind: direction: #{message.weather_wind_direction} speed: #{message.weather_wind_speed} temperature: #{message.weather_temperature} humidity: #{message.weather_humidity} pressure: #{message.weather_pressure} rain: #{message.weather_rain}")
          end
          
          if message.has_course?
            @courses[source] = message
            # logger.warn( "message: course: direction: #{message.course_direction} speed: #{message.course_speed}")
          end
          
          if message.has_phg?
            @phgs[source] = message
            # logger.warn( "message: phg: power: #{message.phg_power} height: #{message.phg_height} gain: #{message.phg_gain} direction: #{message.phg_direction}")
          end
          
        end
      end

      return
    end


    def run 
      logger.info( "run")

      while @thread_run

        begin

          @thread_mutex.synchronize do 

            now = Time.now
            
            # clean up station hash
            @timestamps.each{ |key, timestamp|
              if now - timestamp > @timeout
                delete_station( key)
              end
            }
          
            file = File.new( @file, "w")
          
            # write xml header
            file.puts( "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>")
            
            # write stations element
            stations = sprintf( "<stations call=\"%s\" lat=\"%02.04f\" lng=\"%03.04f\">", @call, @latitude, @longitude)
            file.puts( stations)
            
            # write station element for each station
            @messages.each{ |key, message|
              
              if message

                source = message.source
                source = message.object_address if message.is_object?
                destination = message.destination
                path = message.path
                
                path_list = String.new
                for i in 0...path.length
                  path_list += "," if i != 0 
                  path_list += path[i].to_s
                end
                
                logger.debug( "key: #{key}, source: #{source}")
                
                # start station section
                icon = "else"
                overlay = ""
                
                symbol = @symbols[source]
                if symbol 
                  icon = symbol.symbol_name
                  overlay = symbol.symbol_table
                end
              
                station_open = sprintf( "<station source=\"%s\" destination=\"%s\" path=\"%s\" icon=\"%s\" overlay=\"%s\">", source, destination, path_list, escapeHTML( icon), escapeHTML( overlay))
                file.puts( station_open)
                
                # write position
                position = @positions[source]
                
                output = "<position lat=\"00.0000\" lng=\"000.0000\"/>"
                output = sprintf( "<position lat=\"%02.04f\" lng=\"%03.04f\"/>", position.latitude, position.longitude) if position && position.has_position?
                file.puts( output)
                
                # write timestamp
                timestamp = @timestamps[source]
                if timestamp
                  output = sprintf( "<time year=\"%04.4i\" month=\"%02.2i\" day=\"%02.2i\" hour=\"%02.2i\" minute=\"%02.2i\">", now.year, now.month, now.day, now.hour, now.min)
                  output += sprintf( "%04.4i-%02.2i-%02.2iT%02.2i:%02.2i", now.year, now.month, now.day, now.hour, now.min)
                  output += "</time>"
                  file.puts( output)
                end
                
                # write comment section
                comment = @comments[source]
                if comment && comment.has_comment?
                  value = comment.comment
                  output = sprintf( "<comment>%s</comment>", escapeHTML( value.dump[1..value.length])) 
                  file.puts( output)
                end
                
                # write status section
                status = @states[source]
                if status && status.has_status?
                  value = status.status
                  output = sprintf( "<status>%s</status>", escapeHTML( value.dump[1..value.length])) 
                  file.puts( output)
                end
                
                # write course section
                course = @courses[source]
                if course 
                  output = sprintf( "<course direction=\"%i\" speed=\"%i\"/>", course.course_direction, course.course_speed) if course.has_course?
                  file.puts( output)
                end
                
                # write phg section
                phg = @phgs[source]
                if phg 
                  output = sprintf( "<phg pwr=\"%i\" hgt=\"%i\" gn=\"%i\" dr=\"%s\"/>", phg.phg_power, phg.phg_height, phg.phg_gain, phg.phg_direction) if phg.has_phg?
                  file.puts( output)
                end
                
                # write weather section
                weather = @weathers[source]
                if weather
                  output = sprintf( "<weather windspeed=\"%i\" winddirection=\"%i\" humid=\"%i\" temperature=\"%i\" pressure=\"%f\" rain=\"%i\"/>", weather.weather_wind_speed, weather.weather_wind_direction, weather.weather_humidity, weather.weather_temperature, weather.weather_pressure, weather.weather_rain) if weather.has_weather?
                  file.puts( output)
                end
                
                # end station section
                file.puts( "</station>")
                
              end
              
            }
            
            stations = "</stations>"
            file.puts( stations)
            
            file.close
          end

        rescue Exception => ex
          logger.warn( "ex: #{ex}")
          logger.warn( "backtrace: #{ex.backtrace.join( '\n')}")
        end
          
        sleep @period
        
      end

    end


    # escape html symbols (&/><, copied from cgi class)
    def escapeHTML( value) 
      return value.gsub(/&/n, '&amp;').gsub(/\"/n, '&quot;').gsub(/>/n, '&gt;').gsub(/</n, '&lt;') if value
    end


    def insert_station( new_source, new_distance)
      logger.info( "insert_station( new_source, new_distance)")
      

      # delete old entry
      @distances.each_index{ |i|
        source, distance = @distances[i]
        
        if new_source == source 
          @distances.delete_at( i) 
          break
        end
      }
      
      
      # insert new entry
      inserted = false
      @distances.each_index { |i|
        source, distance = @distances[i]
        
        if distance.nil? || (new_distance && distance > new_distance)
          @distances.insert( i, [new_source, new_distance]) 
          inserted = true 
          break
        end
      }

      @distances.push( [new_source, new_distance]) if inserted == false
      
      logger.log( "insert station: (#{@distances.length}) #{new_source}, #{new_distance}, #{@distances.first[1] if !@distances.empty?} - #{@distances.last[1] if !@distances.empty?}") 

      return
    end


    def delete_station( source, distance = nil)
      logger.info( "delete_station( source, distance)")


      # delete old entry
      @distances.each_index{ |i|
        old_source, old_distance = @distances[i]
        
        if source == old_source
          @distances.delete_at( i) 
          break
        end
      }
      
      
      @timestamps.delete( source)
      @messages.delete( source)
      
      @symbols.delete( source)
      @positions.delete( source)
      @comments.delete( source)
      @states.delete( source)
      @weathers.delete( source)
      @courses.delete( source)
      @phgs.delete( source)
  
      # TODO: why this statement fails sometimes ?
      # logger.log( "delete station: (#{@distances.length}) #{source}, #{distance}, #{@distances.first[1]} - #{@distances.last[1]}")

      return
    end

  end

end
