#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/Plugin'
require 'aprs4r/extensions/Extension'
require 'aprs4r/SmartDigipeaterPluginConfiguration'


module APRS4R

  class SmartDigipeaterPlugin < Plugin
    
    @logger = APRS4RLogger.get_logger( "SmartDigipeaterPlugin")
    

    def initialize
      logger.info( "initialize")

      super

      @new_queue = Queue.new
      @hold_queue = Queue.new
      @extensions = Array.new
      
      @thread_run = true

      return
    end

    
    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @device = configuration.device

      logger.debug( "device: #{@device}")
      logger.debug( "extensions: #{configuration.extensions}")
      
      if configuration.extensions

        begin

          configuration.extensions.each{ |extension_configuration|

            name = extension_configuration.name
            type = extension_configuration.type
            enable = extension_configuration.enable

            logger.debug( "name: #{name}")
            logger.debug( "type: #{type}")
            logger.debug( "enable: #{enable}")
            
            if enable 
              extension = load_extension( name, type)
              
              setup_extension( extension, extension_configuration)

              add_extension( extension)
            end

          }

        rescue Exception => ex
          logger.error( "error loading extenions, ex: #{ex}")
          logger.error( ex.backtrace.join( "\n"))
        end

      end

      logger.warn( "extensions: #{@extensions}")
      
      return
    end
    
    
    def register_listener
      logger.info( "register_listener")

      return if !@enable
      
      @socket_manager.add_listener( @device, self)
      
      return
    end
    
    
    def unregister_listener
      logger.info( "unregister_listener")

      return if !@enable

      @socket_manager.remove_listener( @device, self)
      
      return
    end
    
    
    def start
      logger.info( "start")

      return if !@enable
      
      register_listener
      
      return
    end
    
    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      if message.nil?
        logger.warn( "message: nil")
        return
      end
      
      record = [name, message, 0, false]

      @new_queue.push( record)

      return
    end


    def run
      logger.info( "run")

      while @thread_run

        record = nil
        record = @hold_queue.pop if @hold_queue.length > 0 
        record = @new_queue.pop if record.nil?

        result = handle_record( record)

        if result
          logger.log( "digipeating on #{@device}, message: #{result}")
          @socket_manager.send_message( @device, result)
        end

      end

    end


    def handle_record( record) 
      logger.info( "handle_record( record)")

      return nil if record.nil?
      return nil if !record.is_a?( Array)

      name, message, index, repeat = record

      retrun nil if message.nil?

      for i in index...@extensions.length
        result, message = @extensions[i].handle_message( name, message, repeat)
        
        case result 
          
        when Extension.RESULT_DROP
          return nil
          
        when Extension.RESULT_DELAY
          return nil
          
        when Extension.RESULT_ACCEPT
          # do nothing
        end

      end

      return message
    end



    def load_extension( name, type)
      logger.info( "load_extension( name, type)")

      return nil if type.nil?
      logger.debug( "type: #{type}")
        
      extension = nil
      extension_class = "aprs4r" + "/" + "extensions" + "/" + type
        
      logger.debug( "extension_class: #{extension_class}")
      require extension_class

      logger.debug( "after require")
      
      extension = eval( "#{type}.new")
      logger.debug( "extension: #{extension}")
      
      return extension
    end
       

    def setup_extension( extension, configuration)
      logger.info( "setup_extension( extension, configuration)")

      return if extension.nil? || configuration.nil?

      extension.setup( configuration) 

      return
    end


    def add_extension( extension)
      logger.info( "add_extension( extension)")
      
      @extensions << extension
      
      return
    end

  end

end
