#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSPath'
require 'aprs4r/Plugin'
require 'aprs4r/GatewayPluginConfiguration'


module APRS4R

  class GatewayPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "GatewayPlugin")


    def initialize
      logger.info( "initialize")

      super

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @call = nil
      @call = APRSCall.parse( configuration.call) if configuration.call

      @inDevice = configuration.inDevice
      @outDevice = configuration.outDevice

      @localOnly = false 
      @localOnly = configuration.localOnly if configuration.localOnly

      @replacePath = false 
      @replacePath = configuration.replacePath if configuration.replacePath
      @path = APRSPath.parse( configuration.path)

      if @call.to_s =~ /^MYCALL/i
        raise "GatewayPlugin: plugin: #{@name}, invalid call: #{@call}, please use valid call"
      end

      return
    end


    def start
      logger.info( "start")

      return if !@enable

      register_listener

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      return if !@enable
      
      @socket_manager.add_listener( @inDevice, self)

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      return if !@enable

      @socket_manager.remove_listener( @inDevice, self)

      return
    end

    
    def recv_message( name, message) 
      logger.info( "recv_message( name, message)")

      return if message.nil?

      result = handle_message( name, message)
      
      if result 
        logger.log( "gating from #{name} to #{@outDevice}, message: #{result}")
        @socket_manager.send_message( @outDevice, result)
      end
        
      return
    end


    def handle_message( name, message)
      logger.info( "handle_message( name, message)")

      return nil if message.nil?
      
      # NOGATE check
      return nil if message.path && (APRSPath.has_NOGATE?( message.path) || APRSPath.has_RFONLY?( message.path))

      # local only check
      return nil if @localOnly && !message.is_local?
        
      if @replacePath
        gateway_call = @call.clone
        gateway_call.repeated = true

        gateway_path = APRSPath.clone( @path)

        message.path = []
        APRSPath.append_call( message.path, gateway_call)
        APRSPath.append_path( message.path, gateway_path)
      end

      return message
    end

  end

end
