#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'
require 'aprs4r/KISSDevice'
require 'aprs4r/UDPDevice'
require 'aprs4r/Socket'


module APRS4R

  class AX25Socket < Socket

    @logger = APRS4RLogger.get_logger( "AX25Socket");

    
    def initialize
      logger.info( "initialize")

      super
      
      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @call = configuration.call
      @mode = configuration.mode
      @parameters = configuration.parameters

      if @call.to_s =~ /^MYCALL/i
        raise "AX25Socket: device: #{@name}, invalid call: #{@call}, please use valid call"
      end

      return
    end

    
    def init_socket()
      logger.info( "init_socket")

      mode = @mode

      if mode =~ /^kiss/
        @socket = KISSDevice.new( @parameters, @mode)
      elsif mode =~ /^udp/
        @socket = UDPDevice.new( @parameters, @mode)
      end

      logger.debug( "socket: #{@socket}")

      return
    end


    def read_data
      logger.info( "read_data")

      data = nil

      begin
        data = @socket.read_frame
      rescue Exception => ex
        logger.warn( "read_data: ex: #{ex}")
      end

      return data
    end


    def parse_message( frame)
      logger.info( "parse_message( data)")

      if frame.nil?
        logger.warn( "empty frame received")
        return
      end

      message = APRSMessage.new

      index = 0
      length = frame.length

      
      # read destination callsign 
      if length < index+7
        logger.warn( "no destination callsign")
        return nil
      end

      data = frame[index...index+7]
      destination = read_call( data, false)
      message.destination = destination
      logger.debug( "destination: #{destination}")
      index += 7 
      
      # read source callsign
      if length < index+7
        logger.warn( "no source callsign")
        return nil
      end

      data = frame[index...index+7]
      source = read_call( data, false)
      message.source = source
      logger.debug( "source: #{source}")
      index += 7 
      
      # read repeater path
      message.path = Array.new
      
      while (frame[index-1] & 0x01) == 0
        if length < index+7
          logger.warn( "no path callsigns")
          return nil
        end
        
        data = frame[index...index+7]
        entry = read_call( data)

        if !entry.nil? && !entry.empty?
          message.path << entry
        end

        index += 7
      end
      
      # read protocol fields
      if length < index+2
        logger.warn( "no protocol fields")
        return nil
      end

      # read control field
      control = frame[index]
      logger.debug( "control: #{control.to_i}")
      
      if ((control & 0xE0) == 0) && ((control & 0x03) == 0x03)
        logger.debug( "type: UI-Frame")
      end
      
      # read pid field
      pid = frame[index+1]
      logger.debug( "pid: #{pid.to_i}")
      
      if pid == 0xF0 
        logger.debug( "no layer 3")
      end

      index += 2
      
      
      # read payload
      if length < index
        logger.warn( "no payload")
        return nil
      end

      # drop if payload is to short (#266)
      if length < index + 1
        logger.warn( "payload to short (1 byte)")
        return nil
      end
    
      payload = String.new
      frame[index...frame.length].each{ |value|
        payload << value.chr
      }

      # do not remove trailing crlf (#267)
      message.payload = payload
      # message.payload = payload.chomp if payload
      
      return message
    end


    def serialize_message( message)
      logger.info( "serialize_message( message)")
      
      # logger.debug( "message: #{message.to_s}")

      data = Array.new

      # write destination call
      data.concat( write_call( message.destination, 0x60))

      # write source call
      path = message.path

      if path.length == 0 
        data.concat( write_call( message.source, 0xE1))
      else 
        data.concat( write_call( message.source, 0xE0))
      end
      
      # write path 
      for i in 0...path.length
        call = path[i]

        if i != path.length - 1
          data.concat( write_call( call, 0x60))
        else 
          data.concat( write_call( call, 0x61))
        end
      end

      # write control field
      data.push( ((0xE0 & 0x00) | 0x03))

      # write pid field
      data.push( 0xF0)

      # write payload
      payload = Array.new

      message.payload.each_byte{ |value|
        payload << value
      }
      
      data.concat( payload)

      return data
    end

    
    def write_data( data)
      logger.info( "write_data( data)")

      logger.debug( "port: #{@port}")
      
      @socket.write_frame( data)

      return
    end


    def read_call( data, pathcalls = true)
      logger.info( "read_call( data)")

      call = APRSCall.new

      # parse name
      name = String.new
      for i in 0...6
        value = (data[i] >> 1).chr

        if value != ' '
          name << value
        end
      end
      call.call = name

      # parse ssid
      ssid = (data[6] & 0x1E) >> 1
      call.ssid = ssid

      # FIXME: parse H-bit
      repeated = (data[6] & 0x80)
      if repeated == 0x80 && pathcalls == true
        call.repeated = true
      end

      return call.to_s
    end


    def write_call( callsign, flags)
      logger.info( "write_call( call, flags)")
      
      data = Array.new( 7)
      index = 0
      
      call = APRSCall.parse( callsign)

      # write call
      call.call = call.call + " " * (6-call.call.length)
      call.call.each_byte{ |value|
        data[index] = value << 1
        index+=1
      }
      
      
      # write ssid
      logger.debug( "ssid: #{call.ssid}")
      repeated = 0 
      if call.repeated? 
        repeated = 0x80
      end
      ssid = (repeated | flags | (call.ssid.to_i << 1).to_i)

      data[6] = ssid

      return data
    end

  end

end
