#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RVersion'


module APRS4R

  lock_file = nil


  def APRS4R.lock_lockfile

    begin
      lock_name = ENV['APRS4R_LOCK'] || "/var/lock/aprs4r"
      lock_file = File.new( lock_name, "w") if !File.exist?( lock_name)
      lock_file = File.new( lock_name) if lock_file.nil?
      
      lock_result = lock_file.flock( File::LOCK_EX | File::LOCK_NB)
      
      if lock_result != 0 
        printf( $stderr, "ERROR: another instance of APRS4R is already running.\n")
        throw StandardError.new
      end
      
    rescue StandardError => ex
      exit 8
    rescue Exception => ex
      printf( $stderr, "Error obtaining lock for #{lock_name}, ex: #{ex}, #{ex.class}\n")
      exit 16
    end

    return lock_file
  end


  def APRS4R.unlock_lockfile( lock_file)
    lock_file.flock( File::LOCK_UN) if lock_file
  end

  
  Signal.trap( "TERM") do 
    APRS4R.unlock_lockfile( lock_file)

    version = APRS4RVersion
    version += "-RC" if APRS4RRC == true 

    puts ""
    puts "INFO: shutting down #{APRS4RName} version #{version} pid #{Process.pid}"
    puts ""

    exit 0
  end



  # we want to see stacktraces from all threads
  Thread.abort_on_exception = true


  # create lock file
  lock_file = APRS4R.lock_lockfile

  version = APRS4RVersion
  version += "-RC" if APRS4RRC == true 

  puts ""
  puts "INFO: starting up #{APRS4RName} version #{version} pid #{Process.pid}"
  puts ""


  # check timeout for release candidate
  if APRS4RRC == true
    
    release_date = Time.utc( APRS4RDate[0], APRS4RDate[1], APRS4RDate[2])
    current_date = Time.now

    if current_date < release_date
      puts ""
      puts "WARN: wrong system time, please use time synchronisation"
      puts ""
    else 

      one_day = 24 * 60 * 60
      six_weeks = 6 * 7 * one_day
      
      diff_date = [current_date - release_date, 0].max
      diff_days = (diff_date / one_day).to_i

      # check if release candidate is older than six weeks
      if diff_date > six_weeks
        puts ""
        puts "ERROR: release candidate expired, please update"
        puts "ERROR: software will be stopped now" 
        puts ""
        
        # unlock lock file
        APRS4R.unlock_lockfile( lock_file)
        exit 32
      else
        puts ""
        puts "WARN: you are using a release candidate, please update"
        puts "WARN: software will not start after #{((six_weeks - diff_date)/one_day).to_i} days"
        puts ""
      end
    end
    
  end


  $stdout.flush
  

  # load log4r 
  require 'aprs4r/APRS4RLogger'

  begin
    log4r_name = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/aprs4r.log4r"
    APRS4RLogger.load_yaml_file( log4r_name)
  rescue Exception => ex
    printf( $stderr, "Error loading log4r configuration: #{log4r_name}, ex: #{ex}\n")
    exit 8
  end


  # write pid file 
  begin
    pid_name = ENV['APRS4R_PID'] || "/var/run/aprs4r.pid"
    
    pid_file = File.new( pid_name, "w")
    pid_file.puts( Process.pid)
    pid_file.close
  rescue Exception => ex
    printf( $stderr, "Error writing pid file: #{pid_name}\n")
    exit 4
  end
    

  # load configuration
  require 'aprs4r/APRS4RConfiguration'

  configuration = nil

  begin 
    aprs4r_name = ARGV[0] 
  
    configuration = APRS4RConfiguration.load_configuration( aprs4r_name)
    APRS4RConfiguration.set_configuration( configuration)
  rescue Exception => ex
    printf( $stderr, "Error loading aprs4r configuration: #{aprs4r_name}\n")
    exit 2
  end
  

  # start software
  require 'aprs4r/SocketManager'
  require 'aprs4r/PluginManager'
  

  # init socket manager
  socket_manager = SocketManager.get_instance
  socket_manager.start( configuration)

  sleep 2
  
  # start plugin manager
  plugin_manager = PluginManager.get_instance
  plugin_manager.start( configuration)

#  sleep 10
#
#  printf( "old value: #{configuration.plugins['plugin1'].period}\n")
#  configuration.plugins["plugin1"].mutable = true
#  configuration.plugins["plugin1"].changed = true
#  configuration.plugins["plugin1"].period = 10
#  configuration.plugins["plugin1"].message.payload = ">neue Bake1"
#  
#  printf( "new value: #{configuration.plugins['plugin1'].period}\n")
#
#  plugin_manager.reload
  
  sleep

  # unlock lock file
  APRS4R.unlock_lockfile( lock_file)
  

end

