#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'

require 'aprs4r/Plugin'
require 'aprs4r/BeaconPlugin'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/WeatherBeaconPluginConfiguration'


module APRS4R

  class WeatherBeaconPlugin < BeaconPlugin

    @logger = APRS4RLogger.get_logger( "WeatherBeaconPlugin")


    def initialize
      logger.info( "initialize")

      super

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      return
    end


    def create_beacon
      logger.info( "create_beacon")
      
      plugin = @plugin_manager.get_plugin( WeatherPlugin)

      positionless = false
      positionless = true if @message.payload.nil? || @message.payload.empty?

      weather_payload = nil

      # build aprs weather message
      if plugin

        if !plugin.is_uptodate?( @period)
          logger.warn( "weather data not up to date, not sending weather beacon")
          return nil
        end

        weather_payload = String.new
        
        # wind data
        # degree
        wind_direction = plugin.wind_direction 

        payload = "..."
        payload = sprintf( "%3.3i", wind_direction) if wind_direction
        
        if positionless
          payload = "c..."
          payload = sprintf( "c%3.3i", wind_direction) if wind_direction
        end

        weather_payload += payload

        if !positionless
          weather_payload += "/"
        end
        
        # mph
        wind_avg = WeatherPlugin.mps_to_mph( plugin.wind_avg)

        payload = "..."
        payload = sprintf( "%3.3i", wind_avg) if wind_avg

        if positionless
          payload = "s..."
          payload = sprintf( "s%3.3i", wind_avg) if wind_avg
        end

        weather_payload += payload 
        
        wind_gust = WeatherPlugin.mps_to_mph( plugin.wind_gust)
        payload = "g..."
        payload = sprintf( "g%3.3i", wind_gust) if wind_gust
        weather_payload += payload 

        # temperature 
        # fahrenheit
        temperature = WeatherPlugin.celcius_to_fahrenheit( plugin.temperature)
        payload = "t..."
        payload = sprintf( "t%3.3i", temperature) if temperature
        weather_payload += payload

        # rain
        # hundreths of an inch
        rain_1h = WeatherPlugin.meter_to_inch( plugin.rain_1h)
        payload = "r..."
        payload = sprintf( "r%3.3i", rain_1h * 100.0) if rain_1h
        weather_payload += payload
        
        rain_24h = WeatherPlugin.meter_to_inch( plugin.rain_24h)
        payload = "p..."
        payload = sprintf( "p%3.3i", rain_24h * 100.0) if rain_24h
        weather_payload += payload

        rain_today = WeatherPlugin.meter_to_inch( plugin.rain_today)
        payload = "P..."
        payload = sprintf( "P%3.3i", rain_today * 100.0) if rain_today
        weather_payload += payload

        # humidity 
        # percent
        humidity = plugin.humidity
        payload = "h.."
        payload = sprintf( "h%2.2i", humidity) if humidity
        weather_payload += payload

        # pressure 
        # tenths of millibars/tenths of hPascal
        pressure = plugin.pressure
        payload = "b....."
        payload = sprintf( "b%5.5i", pressure * 10.0) if pressure
        weather_payload += payload
      end


      if !weather_payload.nil?
        message = @message.clone
        logger.debug( "adding weather payload: #{weather_payload}")

        if @message.payload.nil? || @message.payload.empty?
          time = Time.now.utc
          message.payload = time.strftime( "_%m%d%H%M")
        end

        message.payload += weather_payload
      end

      return message
    end

  end

end
