#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Rolf Tschumi <hb9sdb@hb9gl.ch>
#                        Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#
#  SysopPlugin (c) 05. Juli 2008  by Rolf Tschumi <hb9sdb@hb9gl.ch>
#

require 'aprs4r/APRS4RLogger'
require 'aprs4r/APRS4RConfiguration'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/SysopPluginConfiguration'


module APRS4R
  
  class SysopPlugin < Plugin
    
    @logger = APRS4RLogger.get_logger( "SysopPlugin")
    

    def initialize
      logger.info( "initialize")

      super

      @configuration = APRS4RConfiguration.get_configuration
      
      @sysop_mode = false

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")
      
      super( configuration)
      
      @device = configuration.device
      @call = configuration.call

      @sysops = Array.new
      @sysops = configuration.sysops if configuration.sysops

      @password = ""
      @password = configuration.password if configuration.password

      return
    end
    
    
    def start
      logger.info( "start")

      return if !@enable

      register_listener

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      return if !@enable

      @socket_manager.add_listener( @device, self)

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      return if !@enable

      @socket_manager.remove_listener( @device, self)

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      return if message.nil?

      if message.is_message? && message.message_recipient == @call && @sysops.include?( message.source)

        handle_ack( name, message)

        request = message.message_payload

        logger.debug( "request: #{request}")

        if request =~ /^LOGIN/i 
          handle_login( message)
        elsif request =~ /^LOGOUT/i 
          handle_logout( message)
        elsif request =~ /^SYSOP/i 
          handle_sysop( message)
        elsif request =~ /^STATUS/i
          handle_status( message)
        elsif request =~ /^VERSION/i
          handle_version( message)

        elsif @sysop_mode
          
          if request =~ /^LIST/i 
            handle_list( message)
          elsif request =~ /^GET/i
            handle_get( message)
          elsif request =~ /^SET/i
            handle_set( message)
          elsif request =~ /^LOAD/i
            handle_load( message)
          elsif request =~ /^SAVE/i
            handle_save( message)
          elsif request =~ /^RELOAD/i
            handle_reload( message)
          elsif request =~ /^RESTART/i
            handle_restart( message)
          elsif request =~ /^REBOOT/i
            handle_reboot( message)
          end
          
        end
        
      end 
      
      return
    end
    

    def handle_ack( name, message)
      logger.info( "handle_ack( message)")

      # send ack if necessary 
      if name == @device && message.is_message? && message.message_recipient == @call.to_s
        if message.message_ack
          ack = APRSMessage.new( @call, APRS4RCall, ["WIDE2-2"])
          ack.payload = ":" + message.source[0..8].ljust( 9) + ":" + "ack" + message.message_ack
          
          @socket_manager.send_message( @device, ack)
        end
      end
      
    end


    def create_message( recipient, payload)
      logger.info( "create_message()")

      message = APRSMessage.new( @call, APRS4RCall, ["WIDE2-2"])

      message.message_recipient = recipient
      message.message_payload = payload 

      return message
    end
      

    def handle_login( message) 
      logger.info( "handle_login( message)")
      
      command = message.message_payload.strip

      if command =~ /^LOGIN$/i

        @code_index = String.new
        @code_word = String.new

        5.times { 
          index = rand( @password.length)

          @code_index << (index + 1).to_s + " "
          @code_word << @password[index]
        }

        payload = "LOGIN 401 [#{@code_index}] Please send session password (#{@code_word})"
        response = create_message( message.source, payload)

        @socket_manager.send_message( @device, response)

      else 

        tokens = command.split( / /)

        code_word = nil
        code_word = tokens[1] if tokens.length >= 2

        payload = "LOGIN 400 Login was unsuccessful"

        if @code_word == code_word
          @sysop_mode = true
          payload = "LOGIN 200 Login was successful"
        end

        response = create_message( message.source, payload)

        @socket_manager.send_message( @device, response)
      end

      return
    end
    
    
    def handle_logout( message) 
      logger.info( "handle_logout( message)")
      
      @sysop_mode = false
      payload = "LOGOUT 200 Logout was successfull"
      
      response = create_message( message.source, payload)
      
      @socket_manager.send_message( @device, response)
      
      return 
    end


    def handle_status( message) 
      logger.info( "handle_status( message)")

      payload = "STATUS 501 not implemented yet"

      response = create_message( message.source, payload)

      @socket_manager.send_message( @device, response)
      
      return
    end


    def handle_version( message)
      logger.info( "handle_version( message)")

      payload = "VERSION 200 APRS4R-#{APRS4R::APRS4RVersion} Build: #{APRS4R::APRS4RBuild}" 

      response = create_message( message.source, payload)
      
      @socket_manager.send_message( @device, response)
      
      return
    end

   
    def handle_sysop( message)
      logger.info( "handle_sysop( message)")

      payload = "SYSOP 200 sysop mode not active"

      if @sysop_mode
        payload = "SYSOP 200 sysop mode active (be careful)"
      end

      response = create_message( message.source, payload)

      @socket_manager.send_message( @device, response)
      
      return
    end


    def handle_list( message) 
      logger.info( "handle_list( message)")

      tokens = message.message_payload.split( / /)

      path = nil
      path = tokens[1] if tokens.length >= 2

      results = @configuration.list( path)

      results.each{ |result|
        
        payload = "LIST 200 #{result}"

        response = create_message( message.source, payload)
        
        @socket_manager.send_message( @device, response)
      }

      return
    end
    

    def handle_get( message) 
      logger.info( "handle_get( message)")

      tokens = message.message_payload.split( / /)

      path = nil
      path = tokens[1] if tokens.length >= 2
      
      payload = "GET 500 #{path} not readable via remote"
      results = @configuration.get( path)

      if results
        payload = "GET 200 #{path}: "

        results.each{ |result|
          payload += result.to_s
        }
      end

      response = create_message( message.source, payload)
        
      @socket_manager.send_message( @device, response)
      
      return
    end
    
    
    def handle_set( message) 
      logger.info( "handle_set( message)")

      tokens = message.message_payload.split( / /)

      path = nil
      path = tokens[1] if tokens.length >= 2

      value = nil
      value = tokens[2...tokens.length].join( ' ') if tokens.length >= 3

      logger.debug( "path: #{path}")
      logger.debug( "value: #{value}")

      payload = "SET 500 #{path} not writeable via remote"
      result = @configuration.set( path, value)

      if result
        payload = "SET 200 #{path}: "
        results = @configuration.get( path)
        
        results.each{ |result|
          payload += result.to_s
        }
      end

      response = create_message( message.source, payload)
        
      @socket_manager.send_message( @device, response)
      
      return
    end


    def handle_load( message)
      logger.info( "handle_load( message)")

      etc_dir = ENV['APRS4R_ETC'] || "/etc/aprs4r"

      match_data = message.message_payload.match( /^[A-Za-z]+ +(.[^ ]+) *$/i)

      value = match_data[1] if match_data.length == 2

      config_file = File.basename( value)

      logger.debug( "etc_dir: #{etc_dir}")
      logger.debug( "config_file: #{config_file}")

      configuration = APRS4RConfiguration.load_configuration( etc_dir + "/" + config_file)
      logger.debug( "configuration: #{configuration}")

      payload = "LOAD 404 #{config_file} not found"

      if configuration
        APRS4RConfiguration.set_configuration( configuration)

        payload = "LOAD 200 #{config_file} loaded"
      end
      
      response = create_message( message.source, payload)
        
      @socket_manager.send_message( @device, response)

      return
    end


    def handle_save( message)
      logger.info( "handle_save( message)")

      etc_dir = ENV['APRS4R_ETC'] || "/etc/aprs4r"

      match_data = message.message_payload.match( /^[A-Za-z]+ +(.[^ ]+) *$/i)

      value = match_data[1] if match_data.length == 2

      config_file = File.basename( value)

      logger.debug( "etc_dir: #{etc_dir}")
      logger.debug( "config_file: #{config_file}")

      configuration = APRS4RConfiguration.get_configuration
      logger.debug( "configuration: #{configuration}")
      result = APRS4RConfiguration.save_configuration( configuration, etc_dir + "/" + config_file)

      payload = "SAVE 404 #{config_file} failed"
      if result 
        payload = "SAVE 200 #{config_file} saved"
      end
      
      response = create_message( message.source, payload)
        
      @socket_manager.send_message( @device, response)

      return
    end


    def handle_reload( message)
      logger.info( "handle_reload( message)")

      @socket_manager.reload
      @plugin_manager.reload

      payload = "RELOAD 200 successful"

      response = create_message( message.source, payload)
        
      @socket_manager.send_message( @device, response)

      return
    end

    
    def handle_restart( message)
      logger.info( "handle_restart( message)")

      payload = "RESTART 501 not implemented yet"
      
      response = create_message( message.source, payload)

      @socket_manager.send_message( @device, response)

      return
    end


    def handle_reboot( message)
      logger.info( "handle_reboot( message")

      payload = "REBOOT 501 not implemented yet"
      
      response = create_message( message.source, payload)

      @socket_manager.send_message( @device, response)

      return
    end
    
  end
  
end
