#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'thread'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/MessagePluginConfiguration'


module APRS4R

  class MessagePlugin < Plugin

    @logger = APRS4RLogger.get_logger( "MessagePlugin")

    
    def initialize
      logger.info( "initialize")

      super

      @stations = Hash.new
      @mutex = Mutex.new

      @last_expire = Time.now

      @messages = 0

      return
    end

    
    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @rfDevice = configuration.rfDevice
      @isDevice = configuration.isDevice
      @timeout = configuration.timeout

      @path = []
      @path = configuration.path if configuration.path

      return
    end

    
    def start 
      logger.info( "start")

      if ! @enable
        return
      end

      register_listener

      return
    end

    
    def stop
      logger.info( "stop")

      if ! @enable
        return
      end

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      if @enable
        @socket_manager.add_listener( @rfDevice, self)
        @socket_manager.add_listener( @isDevice, self)
      end

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      if @enable
        @socket_manager.remove_listener( @rfDevice, self)
        @socket_manager.remove_listener( @isDevice, self)
      end

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      if message.nil? 
        return
      end

      if name == @rfDevice
        receive_rf_message( message)
      elsif name == @isDevice && message.is_message?
        result = receive_is_message( message)

        if result
          result.path = @path

          logger.debug( "send message: #{result}")
          @socket_manager.send_message( @rfDevice, result)

          @messages += 1
        end

      end

      return
    end


    def receive_rf_message( message)
      logger.info( "receive_rf_message( message)")

      if !message.is_local?
        return
      end
      
      expire_stations
      
      recipient = message.source.to_s
      logger.debug( "recipient: #{recipient} added")
      
      @mutex.synchronize do 
        @stations[recipient] = Time.now
      end
      
      return
    end

    
    def receive_is_message( message)
      logger.info( "receive_is_message( message)")

      if !message.is_message?
        return nil
      end
      
      expire_stations

      sender = message.source.to_s
      recipient = message.message_recipient.to_s
      
      @mutex.synchronize do 

        # is sender in heard list, drop message
        return nil if @stations.has_key?( sender)

        # is recipient in heard list
        if @stations.has_key?( recipient)
          logger.debug( "recipient: #{recipient} found")
          return message
        end
      end

      return nil
    end


    def expire_stations
      logger.info( "expire_stations")

      now = Time.now

      if @last_expire + @timeout < now
        # logger.warn( "before expire: stations.length: #{@stations.length}")

        @mutex.synchronize do 
          # clean up station list 
          @stations.each{ |key, timestamp|
            if now - timestamp > @timeout
              @stations.delete( key)
            end
          }
        end

        # logger.warn( "after expire: stations.length: #{@stations.length}")
        @last_expire = now
      end

      return
    end

    
    def message_count
      return @messages
    end


    def local_count
      return @stations.length
    end

  end

end
