#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Michael Conrad <do5mc@aprs4r.org> and
#                        Andreas Bier <dl1hrc@web.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'socket'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'

require 'aprs4r/Socket'
require 'aprs4r/SocketError'


module APRS4R

  class ISSocket < Socket

    @logger = APRS4RLogger.get_logger( "ISSocket")


    def initialize
      logger.info( "initialize")

      super

      return
    end

    
    def setup( configuration, socket = nil)
      logger.info( "setup( configuration | socket)")
      
      super( configuration)

      @user = "unknown"
      @auth = false

      
      if configuration 
        # socket client mode 
        @clientMode = true 
        
        @hosts = configuration.hosts
        @port = configuration.port
        @username = configuration.username.upcase if configuration.username
        @password = ISSocket.generate_password( @username)
        @filter = configuration.filter
        @timeout = configuration.timeout

        if @username.to_s =~ /^MYCALL/i
          raise "ISSocket: device: #{@name}, invalid call: #{@username}, please use valid call"
        end

      else 
        # socket server mode 
        @enable = true
        @clientMode = false
        @timeout = 0
        @txEnable = true 
        
        @socket = socket
        @socket.nonblock = true
        @socket.sync = true 
      end
      
      logger.debug( "clientMode: #{@clientMode}")
      
      return
    end
    
    
    def init_socket
      logger.info( "init_socket")
      
      if ! @enable 
        return
      end
      
      if @clientMode
        init_client_socket
      else 
        init_server_socket
      end
      
      return
    end
    
    
    def init_client_socket
      logger.info( "init_client_socket")
      
      begin 
        host = @hosts[rand( @hosts.length)]
        
        logger.debug( "connecting to #{host}:#{@port}")
        @socket = TCPSocket.new( host, @port)
        
        @socket.nonblock = true
        @socket.sync = true 
        
        # send login sequence
        username = "user " + @username
        password = "pass " + @password
        software = "vers aprs4r #{APRS4R::APRS4RVersion}-#{APRS4RBuild}"
        software = "vers aprs4r #{APRS4R::APRS4RVersion}-RC-#{APRS4RBuild}" if APRS4R::APRS4RRC == true 
        

        write_data( username + " " + password + " " + software)
        write_data( @filter)
      rescue Exception => ex
        logger.warn( "init_client_socket: ex: #{ex}")
        
        # wait 5 seconds before next try
        sleep 5

        host = @hosts[rand( @hosts.length)]
        
        retry
      end
      
      return
    end
    
    
    def init_server_socket
      logger.info( "init_server_socket")

      software = "\# aprs4r #{APRS4R::APRS4RVersion}"
      software = "\# aprs4r #{APRS4R::APRS4RVersion}-RC" if APRS4R::APRS4RRC == true
      write_data( software)

      @user = "unknown"
      @auth = false

      begin
        Timeout::timeout( 5) do
          line = @socket.gets
          line.chomp! if line

          logger.debug( "start: #{line}")

          match_data = line.match( /# +user +([^ ]+) +pass +([^ ]+)/i)

          logger.debug( "match.length: #{match_data.length}")
          
          if match_data.length >= 3 
            user = match_data[1]
            pass = match_data[2]

            logger.debug( "user: #{user}, pass: #{pass}, generate: #{ISSocket.generate_password( user)}")

            if ISSocket.generate_password( user) == pass
              @user = user 
              @auth = true
            end
          end

        end
      rescue Exception => ex
        logger.warn( "initServerSocket::ex: #{ex}")
      end

      logger.debug( "user: #{@user}, auth: #{@auth}")

      return
    end


    def reinit_socket
      logger.info( "reinit_socket")

      # close socket (with timeout)
      begin
        Timeout::timeout( 5) do
          if ! @socket.nil?
            @socket.close
          end
        end
      rescue Exception => ex
        logger.warn( "reinit_socket: #{ex}")
      end
      
      if @clientMode
        reinit_client_socket
      else 
        reinit_server_socket
      end

      return
    end
    
    
    def reinit_client_socket
      logger.info( "reinit_client_socket")

      init_client_socket
      
      return
    end


    def reinit_server_socket
      logger.info( "reinit_server_socket")

      logger.warn( "server socket reinit")

      # FIXME
      sleep 10

      raise SocketError, "client side down"

      return
    end


    def read_data
      logger.info( "read_data")

      line = nil

      begin
        Timeout::timeout( @timeout) do

          if @socket
            line = @socket.gets.force_encoding( "ASCII-8BIT")
          end

          raise SocketError, "nil message" if line.nil?

          line.chomp! 
          logger.debug( "read_data::line: -#{line}-")

        end

      rescue SocketError
        logger.warn( "read_data: socket ex: #{ex}")
        raise

      rescue Exception => ex
        logger.warn( "read_data: ex: #{ex}")
        # FIXME: insert return nil for benchmark
        # return nil
        reinit_socket 
      end until line !~ /^#/
        
      return line
    end
    

    def parse_message( data)
      logger.info( "parse_message( data)")
      
      logger.debug( "data: #{data}")

      if data.nil?
        return nil
      end
      
      # todo more robust message parsing 

      message = nil
      headerIndex = data.index( ':')
      
      if !headerIndex.nil?
        
        message = APRSMessage.new
        # header
        header = data[0...headerIndex]
        
        sourceIndex = header.index( '>')
        
        if sourceIndex != nil 
          message.source = APRSCall.parse( header[0...sourceIndex]).to_s
        end

        # path
        message.path = Array.new
        path = header[sourceIndex+1...header.length].split( ',')
        message.destination = APRSCall.parse( path[0]).to_s

        path.delete_at( 0)
        
        path.each{ |entry|
          call = APRSCall.parse( entry)
          message.path << call.to_s if call
        }

        # body
        body = data[headerIndex+1...data.length]
        message.payload = body
      end

      return message
    end

    
    def serialize_message( message)
      logger.info( "serialize_message( message)")

      return nil if message.nil?
      
      if message.source.nil? || message.destination.nil? 
        logger.warn( "message: NIL")
        return nil
      end

      data = String.new

      data += message.source.to_s + ">" + message.destination.to_s

      path = message.path

      if !path.nil?
        for i in 0...path.length
          data += "," + path[i].to_s
        end
      end

      data = data + ":" + message.payload

      return data
    end


    def write_data( data)
      logger.info( "write_wata( data)")

      return if data.nil?

      begin
        @socket_mutex.synchronize do
          if @socket 
            logger.debug( "write_data::data: -#{data}-")
            @socket.puts( data + 0x0D.chr)
            @socket.flush
          end
        end
      rescue Exception => ex
        logger.warn( "write_data: ex: #{ex}")
        reinit_socket
      end
      
      return
    end


    def update_filter( filter_calls)
      logger.info( "update_filter( filter_calls)")

      if !filter_calls.nil? && !filter_calls.empty?

        # build new filter command old filter + (b/call1/call2)
        command = @filter + " " + "b"
        filter_calls.each{ |filter|
          command << "/#{filter.upcase}"
        }
        
        # send new filter command
        write_data( command)
      end

      return
    end

    
    def ISSocket.generate_password( value)
      logger.info( "generate_password( value)")

      callsign = APRSCall.parse( value.to_s)
      call = callsign.call

      return if call.nil?

      # init value
      code = 0x73E2
      i = 0

      shift = 8

      call.each_byte{ |value|
        code = code ^ (value << shift)
        
        shift = (shift + 8) % 16
      }

      return (code & 0x7FFF).to_s
    end

  end

end
