#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2009 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'fileutils'

module APRS4R

  # we want to see stacktraces from all threads
  Thread.abort_on_exception = true

  # load log4r 
  require 'aprs4r/APRS4RLogger'
  
  begin
    log4r_name = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/aprs4r.log4r"
    APRS4RLogger.load_yaml_file( log4r_name)
  rescue Exception => ex
    printf( $stderr, "Error loading log4r configuration: #{log4r_name}, ex: #{ex}\n")
    exit 8
  end

  logger = APRS4RLogger.get_logger( "config")


  require 'aprs4r/APRS4RBase'


  # load configuration
  require 'aprs4r/APRS4RConfiguration'

  configuration = nil

  begin 
    old_name = ARGV[0] 
  
    configuration = APRS4RConfiguration.load_configuration( old_name)
    APRS4RConfiguration.set_configuration( configuration)
  rescue Exception => ex
    printf( $stderr, "Error loading aprs4r configuration: #{old_name}\n")
    exit 2
  end
  
  
  # upgrade configuration
  old_version = configuration.version
  old_version = "undefined" if old_version.nil?
  new_version = APRS4R::APRS4RVersion

  logger.debug( "old_version: #{old_version}")
  logger.debug( "new_version: #{new_version}")

  printf( "\n")
  printf( "Starting update of #{old_name} from version #{old_version} to #{new_version}\n")
  printf( "\n")

  update = false

  # update devices
  devices = configuration.devices

  if devices 
    devices.each { |key, device|
      logger.debug( "device: #{device}")
    
      result = device.update_configuration( old_version, new_version)
      
      update = true if result
    }
  end

  # update plugins
  plugins = configuration.plugins
  
  if plugins
    plugins.each{ |key, plugin|
      logger.debug( "plugin: #{plugin}")

      result = plugin.update_configuration( old_version, new_version)
      
      update = true if result
    }
  end

  printf( "update finished\n")

  if update
    new_name = old_name + ".update-backup-#{new_version}"
    FileUtils.cp( old_name, new_name)
  end

  # set new version
  configuration.version = new_version


  # save configuration
  begin 

    APRS4RConfiguration.save_configuration( old_name, configuration)

  rescue Exception => ex
    printf( $stderr, "Error saving aprs4r configuration: \n")
    exit 2
  end


end

