#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'readline'

module APRS4R

  # we want to see stacktraces from all threads
  Thread.abort_on_exception = true

  # load log4r 
  require 'aprs4r/APRS4RLogger'
  
  begin
    log4r_name = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/aprs4r.log4r"
    APRS4RLogger.load_yaml_file( log4r_name)
  rescue Exception => ex
    printf( $stderr, "Error loading log4r configuration: #{log4r_name}, ex: #{ex}\n")
    exit 8
  end

  require 'aprs4r/APRS4RBase'
  require 'aprs4r/APRSMessage'


  class Admin < APRS4RBase
    
    @@local_commands = [
                        'LOCAL', 'REMOTE', 'QUIT'
                       ]
    
    @@remote_commands = [ 
                         'LOGIN', 'LOGOUT', 'VERSION', 'STATUS', 'SYSOP',
                         'LIST', 'GET', 'SET', 
                         'LOAD', 'SAVE', 
                         'RELOAD', 'RESTART', 'REBOOT'
                        ]

    
    @logger = APRS4RLogger.get_logger( "Admin")


    def initialize
      logger.info( "initialize")

      # @thread = Thread.current

      # load configuration
      require 'aprs4r/APRS4RConfiguration'
      
      @configuration = nil
      
      begin 
        aprs4r_name = ARGV[0] 
        
        @configuration = APRS4RConfiguration.load_configuration( aprs4r_name)
      rescue Exception => ex
        printf( $stderr, "Error loading aprs4r configuration: #{aprs4r_name}\n")
        exit 2
      end

      # start socket manager
      require 'aprs4r/SocketManager'

      @device = "is0"
      
      @socket_manager = SocketManager.get_instance
      @socket_manager.start( @configuration)

      @local_station = nil
      @remote_station = nil

      return
    end


    def register_listener
      logger.info( "register_listener")

      @socket_manager.add_listener( @device, self)
      
      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")
      
      # logger.debug( "source: #{message.source}")
      
      if message.is_message? && message.message_recipient == @local_station && message.source == @remote_station
        # logger.debug( "response received: #{message}")

        puts( "#{@remote_station} > " + message.message_payload)

        # @thread.run
      end
      
      return
    end
    

    def command_loop
      logger.info( "command_loop")


      loop do 
        line = Readline::readline( "#{@local_station} > ")
        Readline::HISTORY.push( line)
        
        tokens = line.split( / /)
        
        command = "unknown" 
        command = tokens[0].upcase if tokens.length >= 1
        
        if @@local_commands.include?( command)
          # local commands
          logger.debug( "local commands")
          
          if command == "LOCAL"
            logger.debug( "local")
            
            if tokens.length == 1
              puts( "LOCAL: #{@local_station}")
            elsif tokens.length == 2
              @local_station = tokens[1].upcase 
              puts( "LOCAL: #{@local_station}")
            end
            
          elsif command == "REMOTE"
            logger.debug( "remote")
            
            if tokens.length == 1
              puts( "REMOTE: #{@remote_station}")
            elsif tokens.length == 2
              @remote_station = tokens[1].upcase if tokens.length >= 2
              puts( "REMOTE: #{@remote_station}")
            end
            
          elsif command == "QUIT"
            puts( "bye bye")
            
            exit 0 
          end
          
        elsif @@remote_commands.include?( command)
          # remote commands
          logger.debug( "remote commands")
          
          if @remote_station == "unknown"
            puts( "remote station not set, use REMOTE")
          elsif @local_station == "unknown"
            puts( "local station not set, use LOCAL")
          else
            
            request = APRSMessage.new( @local_station, APRS4R::APRS4RCall, [])
            
            request.message_recipient = @remote_station
            request.message_payload = line
            
            logger.debug( "request message: #{request}")
            
            @socket_manager.send_message( @device, request)

            sleep 2

            # Thread.stop
          end
          
        end
        
      end

    end

  end

  admin = Admin.new

  admin.register_listener
  admin.command_loop

end

