#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'test/unit'

log4r_file = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/default.log4r"

print "log4r_file: #{log4r_file}\n"

require 'aprs4r/APRS4RLogger'

APRS4R::APRS4RLogger::load_yaml_file( log4r_file)

require 'aprs4r/APRSMessage'
require 'aprs4r/GatewayPlugin'
require 'aprs4r/GatewayPluginConfiguration'


module APRS4R 

  class GatewayPluginTest < Test::Unit::TestCase

    @@logger = APRS4RLogger.get_logger( "GatewayPluginTest")
    
    
    def test_default
      configuration = GatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", false, false, [])

      plugin = GatewayPlugin.new
      plugin.setup( configuration)

      # nil message
      nil_message = nil
      result = plugin.handle_message( "in0", nil_message)
      assert_nil( result)

      # nil path 
      nil_path = APRSMessage.new( "FROM", "TO", nil, ">nil path")
      result = plugin.handle_message( "in0", nil_path)
      assert_equal( [], result.path)

      # empty path
      empty_path = APRSMessage.new( "FROM", "TO", nil, ">empty path")
      result = plugin.handle_message( "in0", empty_path)
      assert_equal( [], result.path)

      # rfonly path
      rfonly_path = APRSMessage.new( "FROM", "TO", ["RFONLY"], ">single rfonly")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      rfonly_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "RFONLY", "CALL3*"], ">rfonly in path")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      # nogate path
      nogate_path = APRSMessage.new( "FROM", "TO", ["NOGATE"], ">single nogate")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      nogate_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "NOGATE", "CALL3*"], ">nogate in path")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      # default path
      default_path = APRSMessage.new( "FROM", "TO", ["CALL1*", "CALL2*", "WIDE3-1"], ">default path")
      result = plugin.handle_message( "in0", default_path)
      assert_equal( ["CALL1*", "CALL2*", "WIDE3-1"], result.path)

      return
    end


    def test_replacement
      configuration = GatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", false, true, ["NOGATE", "WIDE2-1"])

      plugin = GatewayPlugin.new
      plugin.setup( configuration)

      # nil message
      nil_message = nil
      result = plugin.handle_message( "in0", nil_message)
      assert_nil( result)

      # nil path 
      nil_path = APRSMessage.new( "FROM", "TO", nil, ">nil path")
      result = plugin.handle_message( "in0", nil_path)
      assert_equal( ["GATEWAY*", "NOGATE", "WIDE2-1"], result.path)

      # empty path
      empty_path = APRSMessage.new( "FROM", "TO", nil, ">empty path")
      result = plugin.handle_message( "in0", empty_path)
      assert_equal( ["GATEWAY*", "NOGATE", "WIDE2-1"], result.path)

      # rfonly path
      rfonly_path = APRSMessage.new( "FROM", "TO", ["RFONLY"], ">single rfonly")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      rfonly_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "RFONLY", "CALL3*"], ">rfonly in path")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      # nogate path
      nogate_path = APRSMessage.new( "FROM", "TO", ["NOGATE"], ">single nogate")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      nogate_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "NOGATE", "CALL3*"], ">nogate in path")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      # default path
      default_path = APRSMessage.new( "FROM", "TO", ["CALL1*", "CALL2*", "WIDE3-1"], ">default path")
      result = plugin.handle_message( "in0", default_path)
      assert_equal( ["GATEWAY*", "NOGATE", "WIDE2-1"], result.path)

      return
    end


    def test_local
      configuration = GatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", true, false, [])

      plugin = GatewayPlugin.new
      plugin.setup( configuration)

      # nil message
      nil_message = nil
      result = plugin.handle_message( "in0", nil_message)
      assert_nil( result)

      # nil path 
      nil_path = APRSMessage.new( "FROM", "TO", nil, ">nil path")
      result = plugin.handle_message( "in0", nil_path)
      assert_not_nil( result)
      assert_equal( [], result.path)

      # empty path
      empty_path = APRSMessage.new( "FROM", "TO", [], ">empty path")
      result = plugin.handle_message( "in0", empty_path)
      assert_not_nil( result)
      assert_equal( [], result.path)

      # rfonly path
      rfonly_path = APRSMessage.new( "FROM", "TO", ["RFONLY"], ">single rfonly")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      rfonly_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "RFONLY", "CALL3*"], ">rfonly in path")
      result = plugin.handle_message( "in0", rfonly_path)
      assert_nil( result)

      # nogate path
      nogate_path = APRSMessage.new( "FROM", "TO", ["NOGATE"], ">single nogate")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      nogate_path = APRSMessage.new( "FROM", "TO", ["CALL1", "CALL2", "NOGATE", "CALL3*"], ">nogate in path")
      result = plugin.handle_message( "in0", nogate_path)
      assert_nil( result)

      # default path
      default_path = APRSMessage.new( "FROM", "TO", ["CALL1*", "CALL2*", "WIDE3-1"], ">default path")
      result = plugin.handle_message( "in0", default_path)
      assert_nil( result)

      return
    end

  end

end
  
