#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2010 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'test/unit'

log4r_file = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/default.log4r"

print "log4r_file: #{log4r_file}\n"

require 'aprs4r/APRS4RLogger'

APRS4R::APRS4RLogger::load_yaml_file( log4r_file)

require 'aprs4r/APRSMessage'
require 'aprs4r/ScheduleBeaconPlugin'
require 'aprs4r/ScheduleBeaconPluginConfiguration'


module APRS4R 

  class ScheduleBeaconPluginTest < Test::Unit::TestCase

    @@logger = APRS4RLogger.get_logger( "ScheduleBeaconPluginTest")
    

    def setup
      @plugin = ScheduleBeaconPlugin.new
      @config = ScheduleBeaconPluginConfiguration.new

      @config.name = "schedule beacon"
      @config.type = "ScheduleBeaconPlugin"
      @config.enable = true
      @config.device = "test0"
      @config.period = 10
      @config.offset = 5
      @config.message = APRSMessage.new( "DO5MC-15", "AP4RXX", ["WIDE1-1", "WIDE2-1"], "> schedule beacon test")

      return
    end


    def set_nil
      @config.startDate = nil
      @config.endDate = nil

      @config.startTime = nil
      @config.endTime = nil

      @config.startDay = nil
      @config.endDay = nil
    end      

    
    def test_nil
      set_nil

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 03, 12, 34)
      time2 = Time.local( 2010, 04, 21,  7, 12)
      time3 = Time.now

      assert_not_nil( @plugin.create_beacon( time1))
      assert_not_nil( @plugin.create_beacon( time2))
      assert_not_nil( @plugin.create_beacon( time3))

      return
    end


    def test_date
      set_nil

      @config.startDate = "2010-04-01"
      @config.endDate = "2010-04-05"

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 03, 12, 34)
      time2 = Time.local( 2010, 04, 21,  7, 12)

      assert_not_nil( @plugin.create_beacon( time1))
      assert_nil( @plugin.create_beacon( time2))

      return
    end


    def test_time
      set_nil

      @config.startTime = "07:00"
      @config.endTime = "15:00"

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 03, 12, 34)
      time2 = Time.local( 2010, 04, 21,  7, 12)
      time3 = Time.local( 2010, 04, 25, 21, 12)

      assert_not_nil( @plugin.create_beacon( time1))
      assert_not_nil( @plugin.create_beacon( time2))
      assert_nil( @plugin.create_beacon( time3))

      return
    end


    def test_day
      set_nil

      @config.startDay = "2"
      @config.endDay = "3"

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 03, 12, 34)
      time2 = Time.local( 2010, 04, 21,  7, 12)
      time3 = Time.local( 2010, 04, 27, 21, 12)

      assert_nil( @plugin.create_beacon( time1))
      assert_not_nil( @plugin.create_beacon( time2))
      assert_not_nil( @plugin.create_beacon( time3))

      return
    end


    def test_day_and_week
      set_nil

      @config.startDay = "2/2"
      @config.endDay = "3/3"

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 07, 12, 34)
      time2 = Time.local( 2010, 04, 13,  7, 12)
      time3 = Time.local( 2010, 04, 17, 21, 12)
      time4 = Time.local( 2010, 04, 21, 21, 12)
      time5 = Time.local( 2010, 04, 22, 21, 12)

      assert_nil( @plugin.create_beacon( time1))
      assert_not_nil( @plugin.create_beacon( time2))
      assert_not_nil( @plugin.create_beacon( time3))
      assert_not_nil( @plugin.create_beacon( time4))
      assert_nil( @plugin.create_beacon( time5))

      return
    end


    def test_date_and_time
      set_nil

      @config.startDate = "2010-04-01"
      @config.endDate = "2010-04-05"

      @config.startTime = "07:00"
      @config.endTime = "15:00"

      @plugin.setup( @config)

      time1 = Time.local( 2010, 04, 03,  7,  0)
      time2 = Time.local( 2010, 04, 03,  8, 34)
      time3 = Time.local( 2010, 04, 03, 15,  0)
      time4 = Time.local( 2010, 04, 25, 12, 34)

      assert_not_nil( @plugin.create_beacon( time1))
      assert_not_nil( @plugin.create_beacon( time2))
      assert_not_nil( @plugin.create_beacon( time3))
      assert_nil( @plugin.create_beacon( time4))

      return
    end

  end

end
  
