#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'io/nonblock'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/WM918WeatherMessage'
require 'aprs4r/WM918WeatherPluginConfiguration'


module APRS4R

  class WM918WeatherPlugin < WeatherPlugin

    @logger = APRS4RLogger.get_logger( "WM918WeatherPlugin")


    def initialize
      logger.info( "initialize")

      super

      return
    end
    

    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @file = configuration.file

      @port = SerialPort.new( @file, 9600, 8, 1, SerialPort::NONE)
      @port.nonblock = true 
      @port.sync = true
      @port.flow_control = SerialPort::SOFT

      return
    end


    def start
      logger.info( "start")

      Thread.new { runWM918Thread }

      return
    end


    def runWM918Thread
      logger.info( "runWM918Thread")
      
      while true 

        begin
          readWeatherData
        rescue Exception => ex
          logger.warn( "runWM918Thread: ex: #{ex}")
        end

      end

      return
    end


    def readWeatherData
      logger.info( "readWeatherData")

      record = readRecord

      if record 
        if record.has_wind?
          logger.info( "wind: direction: #{record.wind_direction}, avg: #{record.wind_avg}, gust: #{record.wind_gust}")
          @wind_direction = record.wind_direction
          @wind_avg = record.wind_avg
          @wind_gust = record.wind_gust
          update_timestamp
        end
        
        if record.has_rain?
          logger.info( "rain: 1h: #{record.rain_1h}, 24h: #{record.rain_24h}, yesterday: #{record.rain_yesterday}")
          @rain_1h = record.rain_1h
          @rain_total = record.rain_24h
          @rain_today = record.rain_today
          update_timestamp
        end
        
        if record.has_thermo?
          logger.info( "thermo: temperature: #{record.temperature}, humidity: #{record.humidity}, dewpoint: #{record.dewpoint}")
          @temperature = record.temperature
          @humidity = record.humidity
          @dewpoint = record.dewpoint
          update_timestamp
        end

        if record.has_baro?
          logger.info( "baro: pressure: #{record.pressure_relative}")
          @pressure = record.pressure_relative
          update_timestamp
        end
      else 
        logger.warn( "empty record received")
      end
      
      return
    end


    def readRecord
      logger.info( "readRecord")

      payload = String.new
      checksum = 0

      # read until HEADER
      value = @port.getc
      while (value & WM918WeatherMessage.WM918_TYPE_MASK) != WM918WeatherMessage.WM918_TYPE_MASK
        value = @port.getc
      end

      type = value 
      logger.debug( "type: #{type.to_s( 16)}")

      length = WM918WeatherMessage.message_length( type)
      logger.debug( "length: #{length}")
      
      payload = Array.new
      if length != nil && length > 2
        (length-2).times { 
          value = @port.getc
          logger.debug( "payload byte: #{value}")
          payload << value
        }
      end

      checksum = @port.getc
      logger.debug( "checksum: #{checksum}")

      # create wm918 weather message
      record = WM918WeatherMessage.new( type, payload, checksum)
      
      if record.is_valid? 
        logger.debug( "message is valid")
        return record
      end
      
      logger.warn( "bad checksum, returning nil")

      return nil
    end

  end

end
