#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Andreas Bier <dl1hrc@web.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'timeout'

require 'aprs4r/APRS4RLogger'
require 'aprs4r/Plugin'
require 'aprs4r/NMEAPluginConfiguration'


module APRS4R

  class NMEAPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "NMEAPlugin")


    def initialize
      logger.info( "initialize")

      super

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @device = configuration.device

      @nmea_device = configuration.nmea_device
      @nmea_baudrate = configuration.nmea_baudrate

      @latitude = configuration.latitude
      @longitude = configuration.longitude

      @distance = configuration.distance

      @nmea_port = nil

      begin 
        @nmea_port = SerialPort.new( @nmea_device, @nmea_baudrate, 8, 1, SerialPort::NONE)
        @nmea_port.nonblock = true
        @nmea_port.flow_control = SerialPort::SOFT
      rescue Exception => ex
        logger.warn( "serial: #{ex}")
      end
      
      return
    end

    
    def start
      logger.info( "start")

      return if !@enable

      register_listener

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      unregister_listener

      return
    end


    def register_listener
      logger.info( "register_listener")

      if @enable
        @socket_manager.add_listener( @device, self)
      end

      return
    end


    def unregister_listener
      logger.info( "unregister_listener")

      if @enable
        @socket_manager.remove_listener( @device, self)
      end

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      return if message.nil?

      return if !message.has_position?
      
      # extract data from incoming message
      # call = APRSCall.parse( message.source).call
      call = message.source
      
      lat = APRSMessage.decimal2latitude( message.latitude)
      lat_ori = lat[lat.length-1].chr
      lat.chop!

      lon = APRSMessage.decimal2longitude( message.longitude)
      lon_ori = lon[lon.length-1].chr
      lon.chop!

      if @distance
        if message.distance( @latitude, @longitude) > @distance
          return
        end
      end

      # wpl header
      nmea_data = "$GPWPL,"
      
      # latitude
      nmea_data << "#{lat.to_s},#{lat_ori},"
      
      # longitude
      nmea_data << "#{lon.to_s},#{lon_ori},"

      # call
      nmea_data << "#{call}"
      
      nmea_csum = 0x00
      for i in 1...nmea_data.length	# Checksumme, nur zwischen $ und *
         nmea_csum = nmea_csum ^ nmea_data[i]
      end
      
      # checksum
      nmea_data << "*"
      nmea_data << sprintf( "%02X", nmea_csum)
      nmea_data << "\r\n"

      logger.debug( "nmea_data: #{nmea_data}")
      
      begin
        if @nmea_port
          @nmea_port.puts( nmea_data)
        end
      rescue Exception => ex
        logger.warn( "serial: #{ex}")
      end

      return
    end


    def position
      logger.info( "position")

      latitude, longitude = nil

      begin
        # we wait 5 seconds for a gprmc nmea sentence
        Timeout::timeout( 5) do

          nmea_data = @nmea_port.gets

          while nmea_data !~ /^\$GPRMC/
            nmea_data = @nmea_port.gets
          end
          
          nmea_values = nmea_data.split( ",")
          
          latitude = APRSCall.latitude2decimal( nmea_values[3].slice( 0..6) + nmea_values[4])
          longitude = APRSCall.longitude2decimal( nmea_values[5].slice( 0..7) + nmea_values[6])
          
        end
        
      rescue Exception => ex
        logger.warn( "serial: #{ex}")
      end

      return [latitude, longitude]
    end

  end

end
