#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Michael Conrad <do5mc@friggleware.de> and
#                        Andreas Bier <dl1hrc@web.de>
#                        Christopher A. Kantarjiev <cak@dimebank.com>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/KISSDeviceConfiguration'


module APRS4R 

  class KISSSerialConnection < APRS4RBase

    KISS_FEND = 0xC0
    KISS_FESC = 0xDB
    KISS_TFEND = 0xDC
    KISS_TFESC = 0xDD

    CHAR_CTRL_C = 0x03
    CHAR_CTRL_M = 0x0D
    CHAR_ESC    = 0x1B

    KISS_EXIT_FRAME = [ KISS_FEND, 0xFF, KISS_FEND, 0x0D]

    @@devices = Hash.new

    @@crc_flex_table = [
                       0x0f87, 0x1e0e, 0x2c95, 0x3d1c, 0x49a3, 0x582a, 0x6ab1, 0x7b38,
                       0x83cf, 0x9246, 0xa0dd, 0xb154, 0xc5eb, 0xd462, 0xe6f9, 0xf770,
                       0x1f06, 0x0e8f, 0x3c14, 0x2d9d, 0x5922, 0x48ab, 0x7a30, 0x6bb9,
                       0x934e, 0x82c7, 0xb05c, 0xa1d5, 0xd56a, 0xc4e3, 0xf678, 0xe7f1,
                       0x2e85, 0x3f0c, 0x0d97, 0x1c1e, 0x68a1, 0x7928, 0x4bb3, 0x5a3a,
                       0xa2cd, 0xb344, 0x81df, 0x9056, 0xe4e9, 0xf560, 0xc7fb, 0xd672,
                       0x3e04, 0x2f8d, 0x1d16, 0x0c9f, 0x7820, 0x69a9, 0x5b32, 0x4abb,
                       0xb24c, 0xa3c5, 0x915e, 0x80d7, 0xf468, 0xe5e1, 0xd77a, 0xc6f3,
                       0x4d83, 0x5c0a, 0x6e91, 0x7f18, 0x0ba7, 0x1a2e, 0x28b5, 0x393c,
                       0xc1cb, 0xd042, 0xe2d9, 0xf350, 0x87ef, 0x9666, 0xa4fd, 0xb574,
                       0x5d02, 0x4c8b, 0x7e10, 0x6f99, 0x1b26, 0x0aaf, 0x3834, 0x29bd,
                       0xd14a, 0xc0c3, 0xf258, 0xe3d1, 0x976e, 0x86e7, 0xb47c, 0xa5f5,
                       0x6c81, 0x7d08, 0x4f93, 0x5e1a, 0x2aa5, 0x3b2c, 0x09b7, 0x183e,
                       0xe0c9, 0xf140, 0xc3db, 0xd252, 0xa6ed, 0xb764, 0x85ff, 0x9476,
                       0x7c00, 0x6d89, 0x5f12, 0x4e9b, 0x3a24, 0x2bad, 0x1936, 0x08bf,
                       0xf048, 0xe1c1, 0xd35a, 0xc2d3, 0xb66c, 0xa7e5, 0x957e, 0x84f7,
                       0x8b8f, 0x9a06, 0xa89d, 0xb914, 0xcdab, 0xdc22, 0xeeb9, 0xff30,
                       0x07c7, 0x164e, 0x24d5, 0x355c, 0x41e3, 0x506a, 0x62f1, 0x7378,
                       0x9b0e, 0x8a87, 0xb81c, 0xa995, 0xdd2a, 0xcca3, 0xfe38, 0xefb1,
                       0x1746, 0x06cf, 0x3454, 0x25dd, 0x5162, 0x40eb, 0x7270, 0x63f9,
                       0xaa8d, 0xbb04, 0x899f, 0x9816, 0xeca9, 0xfd20, 0xcfbb, 0xde32,
                       0x26c5, 0x374c, 0x05d7, 0x145e, 0x60e1, 0x7168, 0x43f3, 0x527a,
                       0xba0c, 0xab85, 0x991e, 0x8897, 0xfc28, 0xeda1, 0xdf3a, 0xceb3,
                       0x3644, 0x27cd, 0x1556, 0x04df, 0x7060, 0x61e9, 0x5372, 0x42fb,
                       0xc98b, 0xd802, 0xea99, 0xfb10, 0x8faf, 0x9e26, 0xacbd, 0xbd34,
                       0x45c3, 0x544a, 0x66d1, 0x7758, 0x03e7, 0x126e, 0x20f5, 0x317c,
                       0xd90a, 0xc883, 0xfa18, 0xeb91, 0x9f2e, 0x8ea7, 0xbc3c, 0xadb5,
                       0x5542, 0x44cb, 0x7650, 0x67d9, 0x1366, 0x02ef, 0x3074, 0x21fd,
                       0xe889, 0xf900, 0xcb9b, 0xda12, 0xaead, 0xbf24, 0x8dbf, 0x9c36,
                       0x64c1, 0x7548, 0x47d3, 0x565a, 0x22e5, 0x336c, 0x01f7, 0x107e,
                       0xf808, 0xe981, 0xdb1a, 0xca93, 0xbe2c, 0xafa5, 0x9d3e, 0x8cb7,
                       0x7440, 0x65c9, 0x5752, 0x46db, 0x3264, 0x23ed, 0x1176, 0x00ff
                      ]
    

    @logger = APRS4RLogger.get_logger( "KISSSerialConnection")

    
    def initialize( configuration, mode)
      logger.info( "initialize( configuration, mode)")
      
      @mode = mode

      return if @mode !~ /^kiss/i 

      @device = configuration.device
      @serial = nil
      @serial_mutex = Mutex.new

      @baudrate = KISSDeviceConfiguration::KISS_DEFAULT_BAUDRATE
      @baudrate = configuration.baudrate.to_i if configuration.baudrate 

      @speed = KISSDeviceConfiguration::KISS_DEFAULT_SPEED
      @speed = configuration.speed if configuration.speed 

      @timeout = KISSDeviceConfiguration::KISS_DEFAULT_TIMEOUT
      @timeout = configuration.timeout.to_i if configuration.timeout


      begin
        # @txdelay = KISSDeviceConfiguration::KISS_DEFAULT_TXDELAY
        @txdelay = configuration.txdelay.to_i if configuration.txdelay
        
        # @persistence = KISSDeviceConfiguration::KISS_DEFAULT_PERSISTENCE
        @persistence = configuration.persistence.to_i if configuration.persistence

        # @slottime = KISSDeviceConfiguration::KISS_DEFAULT_SLOTTIME
        @slottime = configuration.slottime.to_i if configuration.slottime

        @duplex = KISSDeviceConfiguration::KISS_DEFAULT_DUPLEX
        @duplex = true if configuration.duplex

      rescue Exception => ex 
        logger.error( "Error reading configuration: #{ex}")
      end
      
      @ports = Array.new
      @queues = Hash.new
      @thread = nil

      init_serial 

      init_kiss

      thread = Thread.new { run } 

      return
    end


    def KISSSerialConnection.get_device( configuration, mode)
      logger.info( "initialize( configuration)")

      instance = @@devices[configuration.device]
      instance = KISSSerialConnection.new( configuration, mode) if instance.nil?

      if instance.has_port?( configuration.port)
        logger.error( "port already exists")
      else
        instance.add_port( configuration.port)
        instance.set_kiss_parameters( configuration.port)
      end

      return instance
    end


    def init_serial
      logger.info( "init_serial")

      logger.debug( "device: #{@device}")
      logger.debug( "baudrate: #{@baudrate}")

      @serial_mutex.synchronize do 
        begin
          @serial.close if @serial
          @serial = SerialPort.new( @device, @baudrate, 8, 1, SerialPort::NONE)

          @serial.nonblock = true 
          @serial.sync = true

          @serial.flow_control = SerialPort::SOFT
        rescue Exception => ex
          logger.error( "error setting up serial port: #{ex}")
        end
      end

      return
    end


    def init_kiss
      logger.info( "init_kiss")

      mode = @mode.downcase if @mode

      if mode == "kiss/tnc2/northlink"
        init_tnc2_northlink
      elsif mode == "kiss/tnc2/tapr"
        init_tnc2_tapr
      elsif mode == "kiss/tnc2/multi"
        init_tnc2_multi
      elsif mode == "kiss/tnc7"
        init_tnc7
      elsif mode == "kiss/tncx"
        init_tncx
      elsif mode == "kiss/kenwood"
        init_kenwood
      elsif mode == "kiss/kenwood/thd7"
        init_kenwood
      elsif mode == "kiss/kenwood/tmd700"
        init_kenwood
      elsif mode == "kiss/kenwood/tmd710"
        init_kenwood_tmd710
      elsif mode == "kiss/scs/ptc"
        init_scs_ptc
      elsif mode == "kiss/scs/tracker"
        init_scs_tracker
      elsif mode == "kiss/ot2"
        init_ot2
      elsif mode == "kiss/kantronics"
        init_kantronics
      elsif mode == "kiss/noinit"
        logger.info( "no kiss init sequence required")
      else
        init_default
      end

      return
    end


    def reinit_kiss
      logger.info( "reinit_kiss")

      sleep 5

      init_kiss

      return
    end


    def set_kiss_parameters( port)
      logger.info( "set_kiss_parameters( port)")

      # set txdelay
      write_frame( port, [@txdelay], 0x01) if @txdelay
      
      # set persistence
      write_frame( port, [@persistence], 0x02) if @persistence
      
      # set slottime
      write_frame( port, [@slottime], 0x03) if @slottime

      # write dupex
      write_frame( port, [@duplex], 0x04) if @duplex

      return
    end
      


    def init_tnc2_northlink
      logger.info( "init_tnc2_northlink")

      data = String.new
      data << ?\C-X
      data << ?\C-[
      data << "@K"
      data << "\r"

      logger.debug( "send KISS init sequence")
      write_serial( data)
      sleep 1

      return
    end


    def init_tnc2_tapr
      logger.info( "init_tnc2_tapr")
      
      logger.debug( "send KISS exit frame")
      write_serial( KISS_EXIT_FRAME)
      sleep 1
      
      # enable kiss mode
      # logger.debug( "send: mycall #{@call}")
      # write_serial( "mycall #{@call}\r")
      # sleep 1

      # logger.debug( "send: hbaud #{@speed}")
      # write_serial( "hbaud #{@speed}\r")
      # sleep 1

      logger.debug( "send: kiss on ")
      write_serial( "kiss on\r")
      sleep 1

      logger.debug( "send: restart")
      write_serial( "restart\r")
      sleep 1

      return
    end


    def init_tnc2_multi
      logger.info( "init_tnc2_multi")

      init_tnc2_northlink

      return
    end
    

    def init_tnc7
      logger.info( "init_tnc7")
      
      # nothing to do ;-) 

      return
    end


    def init_tncx
      logger.info( "init_tncx")
      
      # nothing to do ;-) 

      return
    end


    def init_scs_ptc
      logger.info( "init_scs_ptc")
      
      # data = String.new
      # data << 0x1B.chr
      write_serial( [ CHAR_ESC ] )
      sleep 1

      # enable packet mode
      logger.debug( "send PACKET command")
      write_serial( "PACKET\r\n")
      sleep 1

      logger.debug( "send: baud #{@speed}")
      write_serial( "baud #{@speed}\r\n")
      sleep 1

      # enable kiss mode
      logger.debug( "send KISS command")
      write_serial( "KISS\r\n")
      sleep 1

      return
    end


    def init_scs_tracker
      logger.info( "init_scs_tracker")

      init_tnc2_northlink

      return
    end


    def init_ot2
      logger.info( "init_ot2")
      
      logger.debug( "send KISS exit frame")
      write_serial( KISS_EXIT_FRAME)
      sleep 1

      logger.debug( "enter kiss mode")
      write_serial( [ CHAR_CTRL_M, CHAR_CTRL_M ] )
      sleep 2

      write_serial( "AMODE KISS\r")
      sleep 2

      return
    end


    def init_kenwood
      logger.info( "init_kenwood")

      logger.debug( "restart kenwood tnc")
      write_serial( "TC 1\r")
      sleep 2
      write_serial( "TC 0\r")
      sleep 2

      # enable software flow control
      logger.debug( "send: TXFLOW ON")
      write_serial( "TXFLOW ON\r")
      sleep 1

      # enable software dcd
      logger.debug( "send: SOFTDCD ON")
      write_serial( "SOFTDCD ON\r")
      sleep 1

      # enable kiss mode
      logger.debug( "send: mycall #{@call}")
      write_serial( "mycall #{@call}\r")
      sleep 1
      
      logger.debug( "send: hbaud #{@speed}")
      write_serial( "hbaud #{@speed}\r")
      sleep 1
      
      logger.debug( "send: kiss on ")
      write_serial( "kiss on\r")
      sleep 1
      
      logger.debug( "send: restart")
      write_serial( "restart\r")
      sleep 1
      
      return
    end


    def init_kenwood_tmd710
      logger.info( "init_kenwood_tmd710")

      logger.debug( "restart kenwood tnc")
      
      # data = String.new
      # data << 0x03.chr
      write_serial( [ CHAR_CTRL_C ] )
      sleep 1

      write_serial( "AI 1\r")
      sleep 2

      write_serial( "TC 1\r")
      sleep 2
      
      write_serial( "TNC 2\r")
      sleep 2

      write_serial( "TC 0\r")
      sleep 2

      write_serial( "\r\r\r")
      sleep 2
      
      logger.debug( "send: mon off ")
      write_serial( "MON OFF\r")
      sleep 1

      logger.debug( "send: kiss on ")
      write_serial( "KISS ON\r")
      sleep 1

      logger.debug( "send: restart")

      write_serial( "RESTART\r")
      sleep 1

      return
    end


    def init_kantronics
      logger.info( "init_kantronics")

      logger.debug( "send INTERFACE KISS command")
      write_serial( "INTERFACE KISS\r\n")
      sleep 1

      logger.debug( "send RESET command")
      write_serial( "RESET\r\n")
      sleep 1
      
      return
    end


    def init_default
      logger.info( "init_default")
      
      # send exit frame 
      logger.debug( "send KISS exit frame")
      write_serial( KISS_EXIT_FRAME)
      sleep 1
      
      # enable kiss mode
      # logger.debug( "send: mycall #{@call}")
      # write_serial( "mycall #{@call}\r")
      # sleep 1

      # logger.debug( "send: hbaud #{@speed}")
      # write_serial( "hbaud #{@speed}\r")
      # sleep 1

      logger.debug( "send: kiss on ")
      write_serial( "kiss on\r")
      sleep 1

      logger.debug( "send: restart")
      write_serial( "restart\r")
      sleep 1

      return
    end

    
    def run
      logger.info( "run")

      while true

        port, frame = read_port_frame
        
        queue = @queues[port]

        if queue && frame
          queue.push( frame)
        end
        
      end

      return
    end

    
    def read_frame( port)
      logger.info( "read_frame( port)")

      logger.debug( "port: #{port}")

      queue = @queues[port]

      logger.debug( "queue: #{queue}")

      return nil if queue.nil?

      return queue.pop
    end

    
    def read_port_frame
      logger.info( "read_port_frame")

      port = nil
      frame = nil

      begin 
        Timeout::timeout( @timeout) do 
          # read until FEND 
          value = read_serial
          while value != KISS_FEND 
            
            raise Exception.new( "Error reading from serial port") if value.nil?
            
            logger.info( "read wire byte: #{value}")
            
            value = read_serial
          end
          
          # read all FENDs
          while value == KISS_FEND
            value = read_serial

            raise Exception.new( "Error reading from serial port") if value.nil?
          end

          # read command byte
          command = value & 0x0F
          port = value & 0xF0
          logger.debug( "command: #{command}, port: #{port}")

          if (command & 0x0F) != 0 
            # logger.warn( "Error: reading invalid command byte from tnc: #{command}")
            return nil
          end
          
          # read frame
          frame = Array.new
          frame << value
          value = read_serial
          while value != KISS_FEND
            frame << value
            value = read_serial
            
            raise Exception.new( "Error reading from serial port") if value.nil?
          end

        end
      rescue Exception => ex
        logger.warn( "readFrame: ex: #{ex}")
        reinit_kiss
        return nil
      end


      data = decode_kiss( frame)

      if data[0] & 0xF0 == 0x20
        # flex crc

        logger.log( "check_crc_flex: #{check_crc_flex( data)}")

        if check_crc_flex( data) && data.length >= 3
          return data[1...data.length-2]
        else
          return nil
        end
      elsif data[0] & 0xF0 == 0x80
        # smack crc
        
        if data.length >= 3
          return data[1...data.length-2]
        end
      end

      return [port, data[1...data.length]]
    end


    def read_serial
      logger.info( "read_serial")

      value = nil

      if RUBY_VERSION =~ /^1\.8\.[0-9]*$/
        value = @serial.getc
      else
        value = @serial.readbyte
      end

      return value
    end


    def write_frame( port, message, command = 0x00)
      logger.info( "write_frame( port, message)")

      begin

        frame = Array.new

        # frame << 0x20
        # frame.concat( message.to_a)
        # crc = calc_crc_flex( frame)
        # frame.concat( crc)

        command += ((port << 4) & 0x70)

        frame << command
        frame.concat( message)

        data = Array.new
        
        data << KISS_FEND
        data.concat( encode_kiss( frame))
        data << KISS_FEND

        # if tnc is an opentracker 2, wait 1/4 second
        if @mode.downcase == "kiss/ot2"
          sleep 0.250
        end
        
        write_serial( data)

      rescue Exception => ex
        logger.warn( "writeFrame::ex: #{ex}")
      end
      
      return
    end


    def write_serial( data)
      logger.info( "write_serial")

      @serial_mutex.synchronize do 
        begin
          
          if data.is_a? String
            @serial.puts( data)
          else
            data.each{ |value|
              # logger.debug( "value.class: #{value.class}, value: #{value}")
              @serial.putc( value)
            }
          end
        rescue Exception => ex
          logger.error( "error writing to serial port: #{ex}")
          logger.error( ex.backtrace.join( "\n"))
        end
      end
        
      return
    end


    def encode_kiss( payload)
      logger.info( "encode_kiss( payload)")

      data = Array.new

      for i in 0...payload.length

        value = payload[i]

        # do kiss encoding (FEND->FESC+TFEND, FESC->FESC+TFESC)
        if value == KISS_FEND
          data << KISS_FESC << KISS_TFEND
        elsif value == KISS_FESC
          data << KISS_FESC << KISS_TFESC
        else 
          data << value
        end

      end

      return data    
    end


    def decode_kiss( payload)
      logger.info( "decode_kiss( payload)")

      if payload.nil?
        logger.warn( "nil payload field")
        return nil
      end

      data = Array.new

      for i in 0...payload.length

        value = payload[i]

        # do kiss decoding (FESC+TFEND->FEND, FESC+TFESC->FESC)
        if value == KISS_FESC
          value = payload[i+=1]
          
          if value == KISS_TFEND
            data << KISS_FEND
          elsif value == KISS_TFESC
            data << KISS_FESC
          end
        else 
          data << value
        end 

      end

      return data
    end


    def has_port?( port)
      logger.info( "has_port?( port)")

      return @ports.include?( port)
    end


    def add_port( port)
      logger.info( "add_port")

      if port
        @ports << port 
        @queues[port] = Queue.new
      end

      return
    end


    def calc_crc_flex( data)
      logger.info( "calc_crc_flex( data)")
      
      result = 0xFFFF
      
      data.each { |value|
        result = ((result << 8) ^ @@crc_flex_table[((result >> 8) ^ value.to_i) & 0xFF]) & 0xFFFF
      }
      
      return [result >> 8, result & 0xF0]
    end
    

    def check_crc_flex( data)
      logger.info( "check_crc_flex( data)")
      
      result = 0xFFFF
      
      data.each { |value|
        result = ((result << 8) ^ @@crc_flex_table[((result >> 8) ^ value.to_i) & 0xFF]) & 0xFFFF
      }
      
      if (result & 0xFFFF) != 0x7070
        return false
      end
      
      return true
    end
    
  end

end
