#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'test/unit'

log4r_file = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/default.log4r"

print "log4r_file: #{log4r_file}\n"

require 'aprs4r/APRS4RLogger'

APRS4R::APRS4RLogger::load_yaml_file( log4r_file)

require 'aprs4r/APRSMessage'
require 'aprs4r/QueryPlugin'
require 'aprs4r/QueryPluginConfiguration'


module APRS4R 

  class QueryPluginTest < Test::Unit::TestCase

    @@logger = APRS4RLogger.get_logger( "QueryPluginTest")
    

    def test_nil_message

      message = nil
    
      configuration = QueryPluginConfiguration.new( "test0", "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)
      
      result = plugin.handle_message( "test0", message)
      assert_nil( result)
    end


    def test_wrong_interface

      message = APRSMessage.new( "FROM", "TO", [], ">no query")
      
      configuration = QueryPluginConfiguration.new( "test0", "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)
      
      result = plugin.handle_message( "test1", message)
      assert_nil( result)
    end      


    def test_no_query
      
      message = APRSMessage.new( "FROM", "TO", [], ">no query")
      
      configuration = QueryPluginConfiguration.new( "test0", "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)
      
      result = plugin.handle_message( "test0", message)
      assert_nil( result)
    end


    def test_general_query

      message = APRSMessage.new( "FROM", "TO", [])
      
      configuration = QueryPluginConfiguration.new( "test0", "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)
      
      # general APRS query
      message.payload = "?APRS?"
      
      result = plugin.handle_message( "test0", message)
      assert( result)
      assert_equal( result.length, 0)

      # general IGATE query
      message.payload = "?IGATE?"
      
      result = plugin.handle_message( "test0", message)
      assert( result)
      assert_equal( result.length, 0)

      # general WX query
      message.payload = "?WX?"
      
      result = plugin.handle_message( "test0", message)
      assert( result)
      assert_equal( result.length, 0)
    end


    def test_APRS
      @@logger.info( "APRS")

      device = "test0"

      position = APRSMessage.new( "ABCDEF", APRS4RCall, [], "!4900.00NI00825.00E&Query-Position")
      status = APRSMessage.new( "ABCDEF", APRS4RCall, [], ">Query-Status")

      query = APRSMessage.new( "FROM", "TO", [], "?APRS?")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, position)
      plugin.recv_local_message( device, status)
      
      # general APRS query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( result.length, 2)


      assert_equal( result[0].to_s, position.to_s)
      assert_equal( result[1].to_s, status.to_s)
    end


    def test_IGATE
      @@logger.info( "IGATE")

      device = "test0"

      station1 = APRSMessage.new( "ABCDEF", APRS4RCall, [], "!4900.00NI00825.00E&IGATE-Query1")
      station2 = APRSMessage.new( "ABCDEF", APRS4RCall, [], "!4900.00NI00826.00E&IGATE-Query2")

      query = APRSMessage.new( "FROM", "TO", [], "?IGATE?")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, position)
      plugin.recv_local_message( device, status)
      
      # general APRS query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( result.length, 2)

      assert_equal( result[0].to_s, position.to_s)
      assert_equal( result[1].to_s, status.to_s)
    end


    def test_WX
      @@logger.info( "WX")

      device = "test0"

      weather = APRSMessage.new( "ABCDEF", APRS4RCall, [], "_10090556c...s...g...t...P012")

      query = APRSMessage.new( "FROM", "TO", [], "?WX?")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, weather)
      
      # general APRS query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( result.length, 1)

      assert_equal( result[0].to_s, weather.to_s)
    end

    
    def test_APRSD
      @@logger.info( "APRSD")

      device = "test0"

      direct1 = APRSMessage.new( "DIRECT1", "UNPROTO", [], ">direct without path")
      direct2 = APRSMessage.new( "DIRECT2", "UNPROTO", [ "DIGI1"], ">direct with unused direct path")
      direct3 = APRSMessage.new( "DIRECT3", "UNPROTO", [ "WIDE1-1"], ">direct with unused general path")
      direct4 = APRSMessage.new( "DIRECT4", "UNPROTO", [ "WIDE1-1", "WIDE3-2"], ">direct with unused large path")

      remote1 = APRSMessage.new( "REMOTE1", "UNPROTO", [ "DIGI1*"], ">remote with used digi path")
      remote2 = APRSMessage.new( "REMOTE2", "UNPROTO", [ "DIGI2*", "WIDE2-1"], ">remote with remaining general path")
      remote3 = APRSMessage.new( "REMOTE2", "UNPROTO", [ "DIGI1", "DIGI2*", "WIDE2-1"], ">remote with remaining general path")

      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.handle_message( device, direct1)
      plugin.handle_message( device, direct2)
      plugin.handle_message( device, direct3)
      plugin.handle_message( device, direct4)
      
      plugin.handle_message( device, remote1)
      plugin.handle_message( device, remote2)
      plugin.handle_message( device, remote3)

      query = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSD")
      
      # APRSD query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)

      return
    end


    def test_APRSH
      @@logger.info( "APRSH")

      device = "test0"

      message1 = APRSMessage.new( "STATION", "TO", [], ">message 1")
      message2 = APRSMessage.new( "STATION", "TO", [], ">message 2")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_message( device, message1)
      plugin.recv_message( device, message2)
      
      query1 = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSH STATION")
      result1 = plugin.handle_message( device, query1)
      @@logger.debug( "result: [#{result1.join( ', ')}]")
      assert( result1)
      assert_equal( 1, result1.length)

      query2 = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSH")
      result2 = plugin.handle_message( device, query2)
      @@logger.debug( "result: [#{result2.join( ', ')}]")
      assert( result2)
      assert_equal( 0, result2.length)

      return
    end


    def test_APRSM
      @@logger.info( "APRSM")

      device = "test0"

      message1 = APRSMessage.new( "STATIONX", "TO", [], ":STATION1 :Hallo")
      message2 = APRSMessage.new( "STATIONY", "TO", [], ":STATION1 :Huhu")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_message( device, message1)
      plugin.recv_message( device, message2)
      
      query1 = APRSMessage.new( "STATION1", "TO", [], ":ABCDEF   :?APRSM")
      result = plugin.handle_message( device, query1)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( result.length, 2)

      query2 = APRSMessage.new( "STATION2", "TO", [], ":ABCDEF   :?APRSM")
      result = plugin.handle_message( device, query2)
      assert( result)
      assert_equal( 0, result.length)

      return
    end


    def test_APRSO
      @@logger.info( "APRSO")

      device = "test0"

      object = APRSMessage.new( "ABCDEF", APRS4RCall, [], ";TEST    *XYZ")

      query = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSO")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, object)
      
      # APRSO query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)

      assert_equal( object.to_s, result[0].to_s)

      return
    end


    def test_APRSP
      @@logger.info( "APRSP")

      device = "test0"

      position = APRSMessage.new( "ABCDEF", APRS4RCall, [], "!4900.00NI00825.00E&Query-Position")

      query = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSP")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, position)
      
      # general APRS query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)

      assert_equal( position.to_s, result[0].to_s)

      return
    end


    def test_APRSS
      @@logger.info( "APRSS")

      device = "test0"

      status = APRSMessage.new( "ABCDEF", APRS4RCall, [], ">Query-Status")

      query = APRSMessage.new( "FROM", "TO", [], ":ABCDEF   :?APRSS")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)

      plugin.recv_local_message( device, status)
      
      # general APRS query
      result = plugin.handle_message( device, query)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)

      assert_equal( status.to_s, result[0].to_s)

      return
    end


    def test_APRST
      @@logger.info( "APRST")

      device = "test0"

      query1 = APRSMessage.new( "FROM1", "TO1", [ "HOP1", "HOP2", "HOP3"], ":ABCDEF   :?APRST")
      query2 = APRSMessage.new( "FROM2", "TO2", [ "HOP4", "HOP5", "HOP6"], ":ABCDEF   :?PING?")
      
      configuration = QueryPluginConfiguration.new( device, "ABCDEF", 300)
      plugin = QueryPlugin.new
      plugin.setup( configuration)
      
      # query1
      result = plugin.handle_message( device, query1)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)
      assert_equal( ":FROM1    :TO1>HOP1,HOP2,HOP3", result[0].payload)

      # query2
      result = plugin.handle_message( device, query2)
      @@logger.debug( "result: [#{result.join( ', ')}]")
      assert( result)
      assert_equal( 1, result.length)
      assert_equal( ":FROM2    :TO2>HOP4,HOP5,HOP6", result[0].payload)

      return
    end


  end

end
  
