#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'test/unit'

log4r_file = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/default.log4r"

print "log4r_file: #{log4r_file}\n"

require 'aprs4r/APRS4RLogger'

APRS4R::APRS4RLogger::load_yaml_file( log4r_file)

require 'aprs4r/APRSMessage'
require 'aprs4r/BuddyGatewayPlugin'
require 'aprs4r/BuddyGatewayPluginConfiguration'


module APRS4R 

  class BuddyGatewayPluginTest < Test::Unit::TestCase

    @@logger = APRS4RLogger.get_logger( "BuddyGatewayPluginTest")
    
    
    def _test_no_filter
      @@logger.info( "no_filter")

      message = APRSMessage.new( "FROM", "TO", [], ">buddy gateway: no filter")
      
      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0")
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)

      result = plugin.handle_message( "in0", message)
      assert_nil( result)
      
      return
    end
    

    def _test_all_filter
      @@logger.info( "all_filter")

      message = APRSMessage.new( "FROM", "TO", [], ">buddy gateway: all filter")
      
      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, ["*"])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)

      
      result = plugin.handle_message( "in0", message)
      assert( result)

      @@logger.debug( "result: #{result}")
      
      return
    end


    def _test_buddy_filter
      @@logger.info( "buddy_filter")

      message1 = APRSMessage.new( "HIT", "TO", [], ">buddy gateway: buddy filter")
      message2 = APRSMessage.new( "MISS", "TO", [], ">buddy gateway: buddy filter")
      
      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, ["HIT"])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      

      result1 = plugin.handle_message( "in0", message1)
      assert( result1)

      result2 = plugin.handle_message( "in0", message2)
      assert_nil( result2)
      
      return
    end


    def _test_buddies_filter
      @@logger.info( "buddies_filter")

      message1 = APRSMessage.new( "BUDDY1", "TO", [], ">buddy gateway test")
      message2 = APRSMessage.new( "BUDDY2", "TO", [], ">buddy gateway test")
      message3 = APRSMessage.new( "BUDDY3", "TO", [], ">buddy gateway test")
      
      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, ["BUDDY1", "BUDDY2"])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      
      
      result1 = plugin.handle_message( "in0", message1)
      assert( result1)

      result2 = plugin.handle_message( "in0", message2)
      assert( result2)
      
      result3 = plugin.handle_message( "in0", message3)
      assert_nil( result3)
      
      return
    end


    def _test_ssid_filter
      @@logger.info( "ssid_filter")

      message1 = APRSMessage.new( "BUDDY1-1", "TO", [], ">buddy gateway test")
      message2 = APRSMessage.new( "BUDDY1-5", "TO", [], ">buddy gateway test")
      message3 = APRSMessage.new( "BUDDY2-2", "TO", [], ">buddy gateway test")
      
      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, ["BUDDY1*"])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      
      
      result1 = plugin.handle_message( "in0", message1)
      assert( result1)

      result2 = plugin.handle_message( "in0", message2)
      assert( result2)
      
      result3 = plugin.handle_message( "in0", message3)
      assert_nil( result3)
      
      return
    end


    def _test_add_message
      @@logger.info( "add_messagge")

      message1 = APRSMessage.new( "CALL-1", "TO", [], ":GATEWAY  :!APRSB ABCDEF")
      message2 = APRSMessage.new( "CALL-1", "TO", [], ":GATEWAY  :?APRSB ABCDEF")
      message3 = APRSMessage.new( "CALL-1", "TO", [], ":GATEWAY  :?APRSB ABCXYZ")

      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, [])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)

      
      result1 = plugin.handle_message( "out0", message1)
      @@logger.debug( "result1: #{result1}")

      result2 = plugin.handle_message( "out0", message2)
      @@logger.debug( "result2: #{result2}")

      result3 = plugin.handle_message( "out0", message3)
      @@logger.debug( "result3: #{result3}")
      
      return
    end


    def _test_send_message
      @@logger.info( "send_message")

      message1 = APRSMessage.new( "CALL-1", "TO", [], ":GATEWAY  :!APRSB ABCDEF")
      message2 = APRSMessage.new( "ABCDEF", "TO", [], "Position report of ABCDEF")

      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, [])
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      

      result1 = plugin.handle_message( "out0", message1)
      @@logger.debug( "result1: #{result1}")

      result2 = plugin.handle_message( "in0", message2)
      assert_equal( message2, result2)
      @@logger.debug( "result2: #{result2}")
      
      return
    end


    def test_timeout
      @@logger.info( "send_message")

      message1 = APRSMessage.new( "CALL-1", "TO", [], ":GATEWAY  :!APRSB ABCDEF")
      message2 = APRSMessage.new( "ABCDEF", "TO", [], "first position report of ABCDEF")
      message3 = APRSMessage.new( "ABCDEF", "TO", [], "second position report of ABCDEF")

      configuration = BuddyGatewayPluginConfiguration.new( "GATEWAY", "in0", "out0", nil, ["BUDDY-1"], 5)
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      
      plugin.update_filter

      result1 = plugin.handle_message( "out0", message1)
      @@logger.debug( "result1: #{result1}")

      result2 = plugin.handle_message( "in0", message2)
      assert_equal( message2, result2)
      @@logger.debug( "result2: #{result2}")

      sleep 10
      
      result3 = plugin.handle_message( "in0", message3)
      assert_nil( result3)

      plugin.update_filter

      return
    end


    def _test_local_heard
      @@logger.info( "local_heard")

      message1 = APRSMessage.new( "CALL-1", "TO", [], ":TEST     :!APRSB ABCDEF")
      message2 = APRSMessage.new( "ABCDEF", "TO", [], "local heard message of ABCDEF")
      message3 = APRSMessage.new( "ABCDEF", "TO", [], "remote message of ABCDEF")

      configuration = BuddyGatewayPluginConfiguration.new( "in0", "out0", [], "TEST")
      plugin = BuddyGatewayPlugin.new
      plugin.setup( configuration)
      

      result1 = plugin.handle_message( "out0", message1)
      @@logger.debug( "result1: #{result1}")

      result2 = plugin.handle_message( "out0", message2)
      assert_nil( result2)

      result3 = plugin.handle_message( "in0", message3)
      assert_nil( result3)
      
      return
    end

  end

end
  
