#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'


module APRS4R

  class WM918WeatherMessage < APRS4RBase

    @@WM918_TYPE_MASK = 0x0F

    @@WM918_MESSAGE_TYPE_HYGRO = 0x8F
    @@WM918_MESSAGE_TYPE_THERMO = 0x9F
    @@WM918_MESSAGE_TYPE_BARO = 0xAF
    @@WM918_MESSAGE_TYPE_RAIN = 0xBF
    @@WM918_MESSAGE_TYPE_WIND = 0xCF


    # message length including preambel and checksum
    @@WM918_MESSAGE_LENGTH = { 

      @@WM918_MESSAGE_TYPE_HYGRO => 35, # humidity sensor data
      @@WM918_MESSAGE_TYPE_THERMO => 34, # thermo sensor data
      @@WM918_MESSAGE_TYPE_BARO => 31, # baro sensor data
      @@WM918_MESSAGE_TYPE_RAIN => 14, # rain sensor data
      @@WM918_MESSAGE_TYPE_WIND => 27 # wind sensor data

    }


    attr_reader :type, :payload, :checksum
    attr_writer :type, :payload, :checksum


    @logger = APRS4RLogger.get_logger( "WM918WeatherMessage")


    def initialize( type, payload, checksum)
      logger.info( "initialize( type, payload, checksum)")

      @type = type
      @payload = payload
      @checksum = checksum

      return
    end


    def is_valid?
      logger.info( "is_valid?")
      
      value = type.to_i
      
      payload.each{ |data|
        value += data.to_i
      }

      if checksum == (value & 0xFF)
        return true
      end
      
      return false
    end


    def has_wind?
      return type == @@WM918_MESSAGE_TYPE_WIND
    end

    def has_rain?
      return type == @@WM918_MESSAGE_TYPE_RAIN
    end

    def has_thermo?
      return type == @@WM918_MESSAGE_TYPE_THERMO
    end

    def has_hygro?
      return type == @@WM918_MESSAGE_TYPE_HYGRO
    end

    def has_baro?
      return type == @@WM918_MESSAGE_TYPE_BARO
    end

    def has_time?
      return type == @@WM918_MESSAGE_TYPE_HYGRO
    end

    def has_date?
      return type == @@WM918_MESSAGE_TYPE_HYGRO
    end


    def wind_direction
      direction = 0.0

      if has_wind? && payload.length >= 6
        direction = high_nibble( payload[5]) * 100.0 + low_nibble( payload[5]) * 10.0 + high_nibble( payload[4])
      end

      return direction
    end


    def wind_avg
      speed = 0.0

      if has_wind? && payload.length >= 5
        speed = low_nibble( payload[4]) * 10.0 + high_nibble( payload[3]) + low_nibble( payload[3]) / 10.0
      end
      
      return speed
    end


    def wind_gust
      speed = 0.0

      if has_wind? && payload.length >= 2
        speed = low_nibble( payload[1]) * 10.0 + low_nibble( payload[0]) + low_nibble( payload[0]) / 10.0
      end

      return speed
    end


    def rain_1h
      rain = 0.0

      if has_rain? && payload.length >= 2
        rain = low_nibble( payload[1]) * 100.0 + high_nibble( payload[0]) * 10.0 + low_nibble( payload[0])

        # convert to m/hr
        rain /= 1000.0
      end

      return rain
    end


    def rain_24h
      rain = 0.0

      if has_rain? && payload.length >= 5
      end

      return rain
    end


    def rain_today
      rain = nil

      return rain
    end


    def rain_yesterday
      rain = 0.0

      if has_rain? && payload.length >= 4
        rain = high_nibble( payload[3]) * 1000.0 + low_nibble( payload[3]) * 100.0 + high_nibble( payload[2]) * 10.0 + low_nibble( payload[2])

        # convert to m/hr
        rain /= 1000.0
      end

      return rain
    end


    def temperature
      temperature = 0.0

      if has_thermo? && payload.length >= 17
        temperature = low_nibble( payload[16].to_i & 0x07) * 10.0 + high_nibble( payload[15]) + low_nibble( payload[15]) / 10.0 

        if (payload[16].to_i & 0x08) == 0x08
          temperature *= -1.0
        end
      end

      return temperature
    end

    
    def humidity
      humidity = 0.0

      if has_hygro? && payload.length >= 21
        humidity = high_nibble( payload[20]) * 10.0 + low_nibble( payload[20])
      end

      return humidity
    end


    def dewpoint
      dewpoint = 0.0
      
      if has_baro? && payload.length >= 18
        dewpoint = high_nibble( payload[17]) * 10.0 + low_nibble( payload[17])
      end
      
      return dewpoint
    end


    def pressure_absolute
      pressure = 0.0

      if has_baro? && payload.length >= 2
        pressure = high_nibble( payload[1]) * 1000.0 + low_nibble( payload[1]) * 100.0 + high_nibble( payload[0]) * 10.0 + low_nibble( payload[0])
      end

      return pressure
    end


    def pressure_relative
      pressure = 0.0

      if has_baro? && payload.length >= 5
        pressure = low_nibble( payload[4]) * 1000.0 + high_nibble( payload[3]) * 100.0 + low_nibble( payload[3]) * 10.0 + high_nibble( payload[2]) + low_nibble( payload[2]) / 10.0
      end
      
      return pressure
    end



    def second
      second = 0

      if has_time? && payload.length >= 1
        second = high_nibble( payload[0]) * 10 + low_nibble( payload[0])
      end

      return second
    end


    def minute
      minute = 0

      if has_time? && payload.length >= 2
        minute = high_nibble( payload[1]) * 10 + low_nibble( payload[1])
      end

      return minute
    end


    def hour
      hour = 0

      if has_time? && payload.length >= 3
        hour = high_nibble( payload[2]) * 10 + low_nibble( payload[2])
      end

      return hour
    end


    def day
      day = 0

      if has_date? && payload.length >= 4
        day = high_nibble( payload[3]) * 10 + low_nibble( payload[3])
      end

      return day
    end


    def month 
      month = 0
      
      if has_date? && payload.length >= 5
        month = low_nibble( payload[4])
      end
      
      return month
    end


    def year 
      return nil
    end
    

    def low_nibble( value)
      return value & 0x0F
    end

    def high_nibble( value)
      return (value & 0xF0) >> 4
    end

    def WM918WeatherMessage.WM918_TYPE_MASK
      return @@WM918_TYPE_MASK
    end

    def WM918WeatherMessage.message_length( type)
      return @@WM918_MESSAGE_LENGTH[type]
    end

  end

end
