#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2009 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/WeatherPlugin'


module APRS4R

  class UltimeterWeatherMessage < APRS4RBase

    attr_reader :payload
    attr_writer :payload


    @@RECORD_WIND_AVG = 1
    @@RECORD_WIND_DIRECTION = 2
    @@RECORD_WIND_GUST = 3

    @@RECORD_TEMPERATURE = 6
    @@RECORD_RAIN_TODAY = 7

    @@RECORD_PRESSURE = 8
    @@RECORD_PRESSURE_DELTA = 9
    @@RECORD_PRESSURE_LSW = 10
    @@RECORD_PRESSURE_MSW = 11

    @@RECORD_HUMIDITY = 13
    @@RECORD_DEWPOINT = 15


    @logger = APRS4RLogger.get_logger( "UltimeterWeatherMessage")


    def initialize( payload)
      logger.info( "initialize( payload)")

      @payload = payload

      return
    end


    def is_valid?
      logger.info( "is_valid?")

      return false if @payload.nil?

      return true if @payload =~ /^\&CR\&/

      return false
    end


    def has_wind?
      return true
    end

    def has_rain?
      return true
    end

    def has_thermo?
      return true
    end

    def has_thermo_indoor?
      return false
    end

    def has_thermo_outdoor?
      return true
    end

    def has_hygro?
      return true
    end

    def has_baro?
      return true
    end

    def has_minute?
      return false
    end

    def has_date?
      return false
    end


    def wind_direction
      direction = nil

      # wind direction 0-255
      value = get_value( @@RECORD_WIND_DIRECTION)

      direction = value * 360.0 / 255.0 if value

      return direction
    end


    def wind_avg
      speed = nil

      # wind avg in 0.1 kph
      value = get_value( @@RECORD_WIND_AVG)
      value /= 10.0 if value

      speed = WeatherPlugin.kph_to_mps( value) if value

      return speed
    end


    def wind_gust
      speed = nil

      # wind gust in 0.1 kph
      value = get_value( @@RECORD_WIND_GUST)
      value /= 10.0 if value 
      
      speed = WeatherPlugin.kph_to_mps( value) if value

      return speed
    end


    def rain_1h
      rain = nil

      return rain
    end

    
    def rain_24h
      rain = nil

      return rain
    end


    def rain_today
      rain = 0.0

      # rain in 0.01 inches
      value = get_value( @@RECORD_RAIN_TODAY)
      value /= 100.0 if value

      rain = WeatherPlugin.inch_to_meter( value) if value

      return rain
    end


    def temperature
      logger.info( "temperature")

      temperature = nil

      # temperature in 0.1 deg F
      value = get_value( @@RECORD_TEMPERATURE)
      logger.debug( "value: #{value}")
      value /= 10.0 if value
      logger.debug( "fahrenheit: #{value}")

      temperature = WeatherPlugin.fahrenheit_to_celcius( value) if value

      logger.debug( "celcius: #{temperature}")

      return temperature
    end

    
    def humidity
      humidity = nil

      # humidity in 0.1%
      value = get_value( @@RECORD_HUMIDITY)

      humidity = value / 10.0 if value

      return humidity
    end


    def dewpoint
      dewpoint = nil

      # temperature in 0.1 deg F
      value = get_value( @@RECORD_DEWPOINT)
      logger.debug( "value: #{value}")
      value /= 10.0 if value
      logger.debug( "fahrenheit: #{value}")

      dewpoint = WeatherPlugin.fahrenheit_to_celcius( value) if value

      logger.debug( "celcius: #{dewpoint}")

      return dewpoint
    end


    def pressure_base
      return nil
    end


    def pressure_absolute
      return nil
    end


    def pressure_relative
      pressure = nil

      # pressure in 0.1 mbar
      value = get_value( @@RECORD_PRESSURE)
      value /= 10.0 if value 

      pressure = value if value

      logger.debug( "barometer value: #{get_value( @@RECORD_PRESSURE)}")
      logger.debug( "barometer delta: #{get_value( @@RECORD_PRESSURE_DELTA)}")
      logger.debug( "barometer lsw: #{get_value( @@RECORD_PRESSURE_LSW)}")
      logger.debug( "barometer msw: #{get_value( @@RECORD_PRESSURE_MSW)}")

      return pressure
    end

    
    def get_value( record_index)
      value = nil

      return nil if @payload.nil?

      index = (record_index.to_i - 1) * 4 + 4

      if @payload.length >= index + 4
        value = @payload[index..index+3]
        logger.debug( "value: #{value}")
        
        return nil if value == "----"

        value = value.hex if value
      else
        logger.warn( "record not found")
      end

      return value
    end

  end

end
