#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'yaml'

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/APRSMessage'

require 'aprs4r/SocketConfiguration'
require 'aprs4r/SampleSocketConfiguration'
require 'aprs4r/ISSocketConfiguration'
require 'aprs4r/AX25SocketConfiguration'
require 'aprs4r/TestSocketConfiguration'

require 'aprs4r/PluginConfiguration'
require 'aprs4r/SamplePluginConfiguration'

require 'aprs4r/BeaconPluginConfiguration'
require 'aprs4r/WeatherBeaconPluginConfiguration'
require 'aprs4r/ProportionalBeaconPluginConfiguration'
require 'aprs4r/StatusBeaconPluginConfiguration'
require 'aprs4r/TelemetryBeaconPluginConfiguration'
require 'aprs4r/ExternalBeaconPluginConfiguration'
require 'aprs4r/ScheduleBeaconPluginConfiguration'

require 'aprs4r/GatewayPluginConfiguration'
require 'aprs4r/BuddyGatewayPluginConfiguration'

require 'aprs4r/DigipeaterPluginConfiguration'
require 'aprs4r/SmartDigipeaterPluginConfiguration'

require 'aprs4r/QueryPluginConfiguration'
require 'aprs4r/MessagePluginConfiguration'

require 'aprs4r/WeatherPluginConfiguration'
require 'aprs4r/DummyWeatherPluginConfiguration'
require 'aprs4r/WMRWeatherPluginConfiguration'
require 'aprs4r/UltimeterWeatherPluginConfiguration'
require 'aprs4r/WS2300WeatherPluginConfiguration'
require 'aprs4r/WM918WeatherPluginConfiguration'

require 'aprs4r/LCD4LinuxPluginConfiguration'
require 'aprs4r/XMLDataPluginConfiguration'
require 'aprs4r/ISServerPluginConfiguration'
require 'aprs4r/SampleTelemetryPluginConfiguration'
require 'aprs4r/LogPluginConfiguration'
require 'aprs4r/StatusPluginConfiguration'
require 'aprs4r/SpeechPluginConfiguration'
require 'aprs4r/WeatherSpeechPluginConfiguration'
require 'aprs4r/NMEAPluginConfiguration'
require 'aprs4r/SysopPluginConfiguration'
require 'aprs4r/MemoryPluginConfiguration'
require 'aprs4r/GreetingPluginConfiguration'

require 'aprs4r/server/T2ServerPluginConfiguration'


module APRS4R 

  class APRS4RConfiguration < APRS4RBase

    attr_accessor :version, :timestamp, :comment, :devices, :plugins

    @logger = APRS4RLogger.get_logger( "APRS4RConfiguration")


    def initialize
      logger.info( "initialize()")

      @devices = Hash.new
      @plugins = Hash.new

      return
    end


    def APRS4RConfiguration.get_configuration()
      logger.info( "get_configuration()")

      return @@configuration 
    end


    def APRS4RConfiguration.set_configuration( configuration)
      logger.info( "set_configuration( configuration)")

      @@configuration = configuration

      return
    end


    def APRS4RConfiguration.load_configuration( name) 
      logger.info( "load_configuration( name)")

      configuration = nil

      begin
        configuration = YAML.load_file( name)
      rescue Exception => ex
        logger.error( "Error reading file: #{name}, ex: #{ex}")
      end

      # if configuration.nil? || configuration == false
      #   configuration = APRS4RConfiguration.new
      # end
        
      return configuration
    end


    def APRS4RConfiguration.save_configuration( name, configuration)
      logger.info( "save_configuration( name, configuration)")

      result = false

      return result if name.nil? || name.empty? || configuration.nil?

      timestamp = Time.now.to_i
      configuration.timestamp = timestamp

      begin
        old_configuration = APRS4RConfiguration.load_configuration( name)

        if old_configuration
          old_configuration.timestamp = timestamp if old_configuration.timestamp.nil?
          old_timestamp = old_configuration.timestamp

          old_name = File.basename( name) 
          old_dir = File.dirname( name)

          File.open( old_dir + "/backups/" + old_timestamp.to_s + "-" + old_name, "w") { |old_file|
            YAML.dump( old_configuration, old_file)
          }
        end
      rescue Exception => ex
        logger.error( "Error creating backup: #{ex}")
      end
        
      begin 
        File.open( name, "w") { |file|
          YAML.dump( configuration, file)
        }
        result = true
      rescue Exception => ex
        logger.error( "Error writing file: #{name}")
      end

      return result
    end


    def list( path = nil)
      logger.info( "list( path)")

      items = Array.new

      if path.nil? || path.empty?
        items << "devices"
        items << "plugins"

      elsif path == "devices"
        @devices.each{ |key, value|

          if value.remote
            name = "devices/#{key}"
            name += "*" if value.changed
            
            items << name
          end
        }

      elsif path == "plugins"
        @plugins.each{ |key, value|

          if value.remote
            name = "plugins/#{key}"
            name += "*" if value.changed
            
            items << name
          end
        }

      elsif path =~ /^devices/
        values = path.split( /\//)

        if !values.nil? && values.length == 2
          name = values[1]

          device = @devices[name]

          if device.remote 
            title = "devices/#{name}"
            title += "*" if device.changed
            
            items << title
            
            if !device.nil?
              device.attributes.each{ |attribute|
                items << "devices/#{name}/#{attribute.name}"
              }
            end

          end

        end

      elsif path =~ /^plugins/
        values = path.split( /\//)

        if !values.nil? && values.length == 2
          name = values[1]
          
          plugin = @plugins[name]

          if plugin.remote
            title = "plugins/#{name}"
            title += "*" if plugin.changed
            
            items << title
            
            if !plugin.nil?
              plugin.attributes.each{ |attribute|
                items << "plugins/#{name}/#{attribute.name}"
              }
            end

          end

        end

      end
          
      return items 
    end

    
    def get( path)
      logger.info( "get( path)")

      items = Array.new

      values = path.split( /\//)

      if !values.nil? && values.length >= 3

        value = nil
        
        if values[0] == "devices"
          value = @devices[values[1]]
          return nil if !value.remote 
        elsif values[0] == "plugins"
          value = @plugins[values[1]]
          return nil if !value.remote
        end

        for i in 2...values.length
          value = value.instance_variable_get( "@#{values[i]}")
        end

        items << value if value
      end

      return items
    end


    def set( path, value) 
      logger.info( "set( path, value)")

      values = path.split( /\//) 

      result = false

      if !values.nil? && values.length >= 3

        object = nil
        
        if values[0] == "devices"
          object = @devices[values[1]]
        elsif values[0] == "plugins"
          object = @plugins[values[1]]
        end

        for i in 2...values.length-1
          object = object.instance_variable_get( "@#{values[i]}")
        end
        
        if object.remote

          attribute_type = object.attribute_type( values.last)
          attribute_value = nil
          
          case attribute_type
            
          when "text"
            attribute_value = value.to_s
          when "number"
            attribute_value = value.to_i
          when "float"
            attribute_value = value.to_f
          when "boolean"
            attribute_value = false
            attribute_value = true if value == "true"
          when "array"
            attribute_value = Array.new
            
            if value 
              value.split( /,/).each{ |entry|
                attribute_value << entry.strip if entry
              }
            end
          when "select"
            attribute_value = value.to_s
          end
          
          object.instance_variable_set( "@#{values.last}", attribute_value)
          object.changed = true

          result = true
        end

      end
      
      return result
    end

    
    def to_yaml_properties
      return %w{@version @timestamp @comment @devices @plugins}
    end

  end

end
