#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2010 by Michael Conrad <do5mc@aprs4r.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RTest'

require 'aprs4r/APRSMessage'

require 'aprs4r/extensions/Extension'
require 'aprs4r/extensions/DigipeaterExtension'
require 'aprs4r/extensions/DigipeaterExtensionConfiguration'


module APRS4R 

  class WideDigipeaterExtensionTest < APRS4RTest

    @logger = APRS4RLogger.get_logger( "WideDigipeaterExtensionTest")


    def setup
      logger.info( "setup")

      configuration = DigipeaterExtensionConfiguration.new
      configuration.name = "wide"
      configuration.type = "DigipeaterExtension"
      configuration.enable = true

      configuration.call = "DIGI1"
      configuration.mode = "wide"
      configuration.aliases = [ "DIGI1", "DIGI2" ]
      configuration.rfonly = true

      @extension = DigipeaterExtension.new
      @extension.setup( configuration)

      @device = "test0"
      @repeat = false

      return
    end


    def test_nil
      logger.info( "test_nil")
      
      message1 = nil

      result, result1 = @extension.handle_message( @device, message1, @repeat)

      assert( result, Extension.RESULT_ACCEPT)
      assert_nil( result1)
    end


    def test_known_alias

      message1 = APRSMessage.new( "FROM", "TO", [ "DIGI1", "WIDE1-1", "WIDE2-1"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI1*", "WIDE1-1", "WIDE2-1"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end


    def test_unknown_alias

      message1 = APRSMessage.new( "FROM", "TO", [ "DIGI5", "WIDE1-1", "WIDE2-1"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI5", "DIGI1*", "WIDE2-1"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end

  
    def test_simple_alias

      message1 = APRSMessage.new( "FROM", "TO", [ "DIGI1", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI1*", "WIDE2-2"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end


    def test_simple_wide1

      message1 = APRSMessage.new( "FROM", "TO", [ "WIDE1-1", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI1*", "WIDE2-2"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end


    def test_simple_wide2

      message1 = APRSMessage.new( "FROM", "TO", [ "DIGI5", "WIDE1*", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = ["DIGI5", "WIDE1*", "DIGI1*", "WIDE2-1"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end


    def test_simple_wide7

      message1 = APRSMessage.new( "FROM", "TO", [ "WIDE7-7"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = [ "DIGI1*", "WIDE7-6"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end


    def test_rfonly

      message1 = APRSMessage.new( "FROM", "TO", [ "RFONLY", "WIDE7-7"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = [ "RFONLY", "DIGI1*", "WIDE7-6"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end

  
    def test_wide1_error
      
      message1 = APRSMessage.new( "FROM", "TO", ["WIDE1", "WIDE2-2"], "")

      result, result1 = @extension.handle_message( @device, message1, @repeat)
      result1_path = [ "WIDE1", "DIGI1*", "WIDE2-1"]

      assert( result, Extension.RESULT_ACCEPT)
      assert( result1)
      assert_equal( result1_path, result1.path)
    
      return
    end

  end

end
  
