#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2008 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'aprs4r/Plugin'


module APRS4R

  class SpeechPlugin < Plugin

    @logger = APRS4RLogger.get_logger( "SpeechPlugin")


    def initialize
      logger.info( "initialize")

      super

      @thread_run = false
      @thread = nil

      return
    end


    def setup( configuration)
      logger.info( "setup( configuration)")

      super( configuration)

      @period = configuration.period
      @offset = configuration.offset

      @text = configuration.text

      @command = configuration.command
      @path = configuration.path

      return
    end


    def start
      logger.info( "start")

      return if !@enable

      @thread_run = true
      @thread = Thread.new{ run }

      return
    end

    
    def stop
      logger.info( "stop")

      return if !@enable

      return
    end


    def run
      logger.info( "run")

      sleep @offset

      while @thread_run
        
        text = create_text
        words = create_words( text)
        play_words( words)

        sleep @period
        
      end
      
      return
    end


    def create_text
      logger.info( "create_text")

      return "no text configured"
    end


    def create_words( text)
      logger.info( "process_words( text)")

      words = Array.new

      items = text.split( / /)
      items.each { |item|

        if is_word?( item)
          words.concat( process_word( item))
        elsif is_number?( item)
          words.concat( process_number( item))
        else
          words << "_beep_"
        end

        words << "_pause_"
      }

      return words
    end


    def is_word?( value)
      return value =~ /^[a-zA-Z]+$/
    end


    def is_number?( value)
      return value.to_s =~ /^[\d.\+\-]+$/
    end
    

    def process_word( word)
      return [word.strip.downcase]
    end


    def process_number( number)
      value = number.to_i

      digits = Array.new

      if value < 0 
        digits << "minus"
        value = value.abs
      end

      # 10e4
      value %= 10000
      digits << ((value / 1000).to_s + "000") if value >= 1000

      # 10e3
      value %= 1000
      digits << ((value / 100).to_s + "00") if value >= 100
    
      # 10e2
      value %= 100

      if value <= 20 
        digits << value.to_s
      else
        one_digit = value % 10 
        ten_digit = value - one_digit
        one_digit = "ein" if one_digit == 1

        if one_digit == 0 
          digits << ten_digit.to_s
        else
          digits.concat( [one_digit.to_s, "und", ten_digit.to_s])
        end

      end

      if number.to_i != number.to_f
        value = number.to_f.abs - number.to_i.abs
        
        digits << "komma"
        
        value *= 10.0
        digits << value.to_i.to_s

        value -= value.to_i
        value *= 10.0
        digits << value.to_i.to_s
      end

      return digits
    end
    

    def play_words( words)
      logger.info( "play_words( words)")

      command = @command

      logger.debug( "command: #{command}")

      words.each{ |word|
        logger.debug( "word: #{word}, #{word.class}")
        command += " " + @path + "/" + word + ".wav"
      }

      logger.debug( "command: #{command}")

      begin 
        system( command)
      rescue Exception => ex
        logger.error( "system: #{ex}")
      end
      
      return
    end

  end

end
