#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/SocketRecvThread'
require 'aprs4r/SocketSendThread'


module APRS4R

  class SocketManager < APRS4RBase

    @logger = APRS4RLogger.get_logger( "SocketManager")

    @@instance = nil


    def initialize
      logger.info( "initialize")

      @recvThreads = Hash.new
      @sendThreads = Hash.new

      @sockets = Hash.new
      @recv_listeners = Hash.new
      @local_listeners = Hash.new
      @duplicates = Hash.new

      @recvCount = Hash.new
      @sendCount = Hash.new
      @lastMessage = nil

      return
    end


    def SocketManager.get_instance
      logger.info( "get_instance")
      
      @@instance = SocketManager.new if @@instance.nil?

      return @@instance
    end


    def start( configuration)
      logger.info( "start( configuration)")

      sockets = configuration.devices
      
      return if sockets.nil?

      sockets.each { |name, configuration|
        
        begin
          
          logger.debug( "configuration: #{configuration}")
          
          name = configuration.name
          type = configuration.type
          enable = configuration.enable
          
          if enable 
            logger.debug( "loading socket: #{type}")
            socket = load_socket( type)

            logger.debug( "setting up socket: #{socket}")
            setup_socket( socket, configuration)

            logger.debug( "starting socket: #{socket}")
            start_socket( socket, name)
          end

        rescue Exception => ex
          logger.error( "socket initialisation failed: name: #{name}, ex: #{ex}")
          logger.error( ex.backtrace.join( "\n"))
        end
      
      }
        
      return
    end


    def reload
      logger.info( "reload")

      logger.warn( "RELOAD not implemented YET")

      return
    end


    def load_socket( type)
      logger.info( "load_socket( type)")

      return nil if type.nil?

      socket = nil
      socket_class = "aprs4r" + "/" + type 

      logger.debug( "socket_class: #{socket_class}")
      require socket_class

      socket = eval( "#{type}.new")
      logger.debug( "socket: #{socket}")

      return socket
    end


    def setup_socket( socket, configuration)
      logger.info( "setup_socket( socket, configuration)")

      return if socket.nil? || configuration.nil?

      socket.setup( configuration) if socket

      return
    end

    
    def start_socket( socket, name)
      logger.info( "start_socket( socket, name)")

      return if socket.nil?

      # start recv thread
      recvThread = SocketRecvThread.new( socket)
      @recvThreads[name] = recvThread
      
      Thread.new { 
        recvThread.run
      }
      
      logger.debug( "recv thread: #{recvThread}")
      
      # start send thread
      sendThread = SocketSendThread.new( socket)
      @sendThreads[name] = sendThread
      Thread.new { 
        sendThread.run
      }
      
      logger.debug( "send thread: #{sendThread}")
              
      # start packet handle thread
      Thread.new { 
        run( name)
      }
              
      @sockets[name] = socket
      @recv_listeners[name] = Array.new
      @local_listeners[name] = Array.new
      @duplicates[name] = Hash.new
      
      @recvCount[name] = 0
      @sendCount[name] = 0
      
      return
    end


    def stop_socket( socket)
      logger.info( "stop_socket( socket)")

      logger.error( "not implemented yet...")
    end


    def run( name)
      logger.info( "run( name)")

      recvThread = @recvThreads[name]

      while true 
        message = recvThread.recv_message

        recv_message( name, message)
      end

    end

    
    def add_listener( name, listener)
      logger.info( "add_listener( name, listener)")
      
      add_recv_listener( name, listener)
    end


    def add_recv_listener( name, listener)
      logger.info( "add_recv_listener( name, listener)")
      
      if @sockets.key?( name) 
        listeners = @recv_listeners[name]
        listeners << listener
      end

      return
    end


    def remove_listener( name, listener)
      logger.info( "remove_listener( name, listener)")

      remove_recv_listener( name, listener)
    end


    def remove_recv_listener( name, listener)
      logger.info( "remove_recv_listener( name, listener)")

      if @sockets.key?( name)
        listeners = @recv_listeners[name]
        listeners.delete( listener)
      end

      return
    end


    def add_local_listener( name, listener)
      logger.info( "add_local_listener( name, listener)")

      if @sockets.key?( name)
        listeners = @local_listeners[name]
        listeners << listener
      end

      return
    end


    def remove_local_listener( name, listener)
      logger.info( "remove_local_listener( name, listener)")

      if @sockets.key?( name)
        listeners = @local_listeners[name]
        listeners.delete( listener)
      end

      return
    end


    def recv_message( name, message)
      logger.info( "recv_message( name, message)")

      return if message.nil?

      if duplicate?( name, message)
        logger.log( "recv: duplicate message dropped")
        return
      end

      @recvCount[name] += 1
      @lastMessage = message
      
      # deliver message to all registered listeners
      listeners = @recv_listeners[name]
      
      if ! listeners.nil?
        listeners.each{ |listener|
          begin 
            listener.recv_message( name, message.clone)
          rescue Exception => ex 
            logger.error( "recv_message failed for plugin: #{listener}, ex: #{ex}")
            logger.error( ex.backtrace.join( "\n"))
          end
        }
      end

      return
    end

    
    def send_message( name, message)
      logger.info( "send_message( name, message)")
      
      sendThread = @sendThreads[name]

      return if message.nil?

      if sendThread.nil?
        logger.warn( "device: #{name} not found")
        return
      end

      if duplicate?( name, message)
        logger.log( "send: duplicate message dropped (from #{message.source})")
        return
      end

      @sendCount[name] += 1

      # send message and wakeup waiting thread
      sendThread.send_message( name, message)

      add_duplicate( name, message)

      return
    end


    def send_local_message( name, message)
      logger.info( "send_local_message( name, message)")

      return if message.nil?

      # deliver message to all registered listeners
      listeners = @local_listeners[name]

      if ! listeners.nil?
        listeners.each{ |listener|
          begin 
            listener.recv_local_message( name, message.clone)
          rescue Exception => ex
            logger.error( "recv_local_message failed for plugin: #{listener}, ex: #{ex}")
            logger.error( ex.backtrace.join( "\n"))
          end
        }
      end

      return
    end


    def update_filter( name, filter)
      logger.info( "update_filter( filter)")

      socket = @sockets[name]

      begin
        socket.update_filter( filter) if socket
      rescue Exception => ex
        logger.error( "filter update failed: #{ex}")
        logger.error( ex.backtrace.join( "\n"))
      end

      return
    end


    def add_duplicate( name, message, now = Time.now)
      logger.info( "add_duplicate( name, message, now)")

      duplicatePeriod = 0

      socket = @sockets[name]

      if !socket.nil?
        duplicatePeriod = socket.duplicatePeriod
      end

      duplicates = @duplicates[name]

      if duplicatePeriod
        # remove old entries from duplicate list
        duplicates.each{ |key, timestamp|
          if now - timestamp > duplicatePeriod
            duplicates.delete( key)
          end
        }
      end
      
      # add message to duplicate list
      duplicates[message.duplicate_key] = now

      return
    end


    def duplicate?( name, message, now = Time.now)
      logger.info( "duplicate?( name, message, now)")
      
      duplicatePeriod = 0

      socket = @sockets[name]

      if !socket.nil?
        duplicatePeriod = socket.duplicatePeriod
      end

      # duplicate check (if enabled)
      if duplicatePeriod 
        duplicates = @duplicates[name]
        
        timestamp = duplicates[message.duplicate_key]
        
        # drop if timestamp exists and message is a duplicate
        if timestamp && (now - timestamp < duplicatePeriod)
          return true
        end
      end

      return false
    end

    
    def recv_count( name)
      return @recvCount[name]
    end

    def send_count( name)
      return @sendCount[name]
    end

    def last_message
      return @lastMessage
    end

  end

end
