#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'test/unit'

log4r_file = ENV['APRS4R_LOG4R'] || "/etc/aprs4r/default.log4r"

print "log4r_file: #{log4r_file}\n"

require 'aprs4r/APRS4RLogger'

APRS4RLogger::load_yaml_file( log4r_file)

require 'aprs4r/APRSMessage'
require 'aprs4r/DigipeaterPlugin'
require 'aprs4r/DigipeaterPluginConfiguration'


class DigipeaterPluginTest < Test::Unit::TestCase

  
  def setup
    configuration = DigipeaterPluginConfiguration.new

    configuration.device = "test0"
    configuration.call = "DIGI"
    configuration.mode = "wide"
    configuration.aliases = [ "DIGI"]
    configuration.rfonly = true
    configuration.reducePath = true
    configuration.permittedHops = 1

    @plugin = DigipeaterPlugin.new( configuration)
  end

  
  def _test_translate

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["RELAY", "WIDE", "WIDE", "TRACE", "TRACE4-3"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.translate_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_translate_used_relay
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["RELAY*", "WIDE*", "WIDE", "TRACE", "TRACE4-3"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.translate_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_optimize_wide
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE2-2", "WIDE3-3", "WIDE4-4"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.optimize_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_optimize_trace
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "TRACE2-2", "TRACE3-3", "TRACE4-4"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.optimize_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_optimize_wide_trace
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE2-2", "WIDE3-3", "TRACE3-3", "TRACE2-2"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.optimize_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_reduce_wide_small
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE2-2"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.reduce_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_reduce_wide
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE5-5", "WIDE4-4"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.reduce_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_reduce_wide_small
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE2-1", "WIDE2-1"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.reduce_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_reduce_trace
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "TRACE5-5", "TRACE3-3"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.reduce_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_reduce_wide_trace
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = ["WIDE1-1", "WIDE5-5", "TRACE6-6"]
    message.payload = ""

    print "message: #{message}\n"

    @plugin.reduce_path( message)

    print "message: #{message}\n"

    return
  end


  def _test_digipeater_none

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = Array.new
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert_nil( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_relay

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "RELAY"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_relay_wide

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "RELAY", "WIDE", "WIDE"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_relay_wide1

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "RELAY", "WIDE1-1"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_wrong_relay

    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "DO5MC", "RELAY*", "WIDE"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_known_alias
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "DO5MC", "WIDE1-1", "WIDE2-1"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_digipeater_unknown_alias
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "DL1LJ", "WIDE1-1", "WIDE2-1"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end

  
  def _test_do_digipeating_simple_alias
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "DO5MC", "WIDE2-2"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_do_digipeating_simple_wide1
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "WIDE1-1", "WIDE2-2"]
    message.payload = ""

    print "message: #{message}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result}\n"
    
    return
  end


  def _test_do_digipeating_simple_wide2
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "DL1LJ", "WIDE1*", "WIDE2-2"]
    message.payload = ""

    print "message: #{message.to_s}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result.to_s}\n"
    
    return
  end


  def _test_do_digipeating_simple_wide7
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "RELAY", "WIDE7-7", "WIDE7-7"]
    message.payload = ""

    print "message: #{message.to_s}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result.to_s}\n"
    
    return
  end


  def _test_do_digipeating_rfonly
    message = APRSMessage.new
    message.source = "DO5MC"
    message.destination = "AP4R00"
    message.path = [ "RFONLY", "WIDE1-1", "WIDE7-7", "WIDE7-7"]
    message.payload = ""

    print "test: do_digipeating_rfonly\n"
    print "message: #{message.to_s}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)

    print "result: #{result.to_s}\n"
    
    return
  end

  
  def _test_do_digipeating_wide1_error
    message = APRSMessage.new( "DO5MC", "AP4R09", ["WIDE1", "WIDE2-2"], "")

    print "test: do_digipeating_wide1_error\n"
    print "message: #{message.to_s}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)
    
    print "result: #{result.to_s}\n"

    return
  end


  def test_do_digipeating_substitution_error
    message = APRSMessage.new( "DO5MC", "APU25N", ["APRS", "WIDE1-1", "WIDE2-2"], "")

    print "test: do_digipeating_substitution_error\n"
    print "message: #{message.to_s}\n"

    result = @plugin.do_digipeating( "test0", message)

    assert( result)
    
    print "result: #{result.to_s}\n"

    return
  end


  def test_do_post_check_relay
    message1 = APRSMessage.new( "DO5MC", "APU25N", ["RELAY", "DF0UK*"], "")

    print "message1: #{message1.to_s}\n"
    result1 = @plugin.do_post_check( message1)
    print "result1: #{result1.to_s}\n"

    assert( result1.path.length == 1)


    message2 = APRSMessage.new( "DO5MC", "APU25N", ["DF0UK*", "RELAY"], "")

    print "message2: #{message2.to_s}\n"
    result2 = @plugin.do_post_check( message2)
    print "result2: #{result2.to_s}\n"

    assert( result2.path.length == 1)
    
    return
  end


  def test_do_post_check_wide1
    message1 = APRSMessage.new( "DO5MC", "APU25N", ["WIDE1-1", "DF0UK*"], "")

    print "message1: #{message1.to_s}\n"
    result1 = @plugin.do_post_check( message1)
    print "result1: #{result1.to_s}\n"

    assert( result1.path.length == 1)


    message2 = APRSMessage.new( "DO5MC", "APU25N", ["DF0UK*", "WIDE1-1"], "")

    print "message2: #{message2.to_s}\n"
    result2 = @plugin.do_post_check( message2)
    print "result2: #{result2.to_s}\n"

    assert( result2.path.length == 1)
    
    return
  end


  def test_do_post_check_widen
    message1 = APRSMessage.new( "DO5MC", "APU25N", ["DF0UK*", "WIDE2-1"], "")

    print "message1: #{message1.to_s}\n"
    result1 = @plugin.do_post_check( message1)
    print "result1: #{result1.to_s}\n"

    assert( result1.path.length == 2)


    message2 = APRSMessage.new( "DO5MC", "APU25N", ["DF0UK*", "WIDE3-3"], "")

    print "message2: #{message2.to_s}\n"
    result2 = @plugin.do_post_check( message2)
    print "result2: #{result2.to_s}\n"

    assert( result2.path.length == 2)
    
    return
  end


end
  
