#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RLogger'

require 'thread'

require 'aprs4r/APRSMessage'
require 'aprs4r/Plugin'
require 'aprs4r/MessagePluginConfiguration'


module APRS4R

  class MessagePlugin < Plugin

    @logger = APRS4RLogger.get_logger( "MessagePlugin")

    
    def initialize( configuration)
      logger.info( "initialize( configuration)")

      super( configuration)

      @rfDevice = configuration.rfDevice
      @isDevice = configuration.isDevice
      @timeout = configuration.timeout
      @path = configuration.path

      @stations = Hash.new
      @mutex = Mutex.new
      
      return
    end

    
    def start 
      logger.info( "start")

      if ! @enable
        return
      end

      registerListener

      return
    end

    
    def registerListener
      logger.info( "registerListener")

      if @enable
        SocketManager.addListener( @rfDevice, self)
        SocketManager.addListener( @isDevice, self)
      end

      return
    end


    def unregisterListener
      logger.info( "registerListener")

      if @enable
        SocketManager.removeListener( @rfDevice, self)
        SocketManager.removeListener( @isDevice, self)
      end

      return
    end


    def receiveAPRSMessage( name, message)
      logger.info( "receiveAPRSMessage( name, message)")

      if message.nil? 
        return
      end

      if name == @rfDevice
        receiveRFAPRSMessage( message)
      elsif name == @isDevice && message.is_message?
        message = receiveISAPRSMessage( message)

        if message
          message.path = []
          message.path = @path if @path 

          logger.debug( "send message: #{message}")
          SocketManager.sendAPRSMessage( @rfDevice, message)
        end

      end

      return
    end


    def receiveRFAPRSMessage( message)
      logger.info( "receiveRFAPRSMessage( message)")

      if !message.is_local?
        return
      end
      
      expireStations
      
      now = Time.now
      key = message.source.to_s

      logger.debug( "station: #{key} added")
      
      @mutex.synchronize do 
        @stations[key] = now
      end
      
      return
    end

    
    def receiveISAPRSMessage( message)
      logger.info( "receiveISAPRSMessage( message)")

      expireStations

      if !message.is_message?
        return nil
      end
      
      key = message.message_recipient
      
      # TODO: check if sender is not in heard list 
      @mutex.synchronize do 
        if @stations.has_key?( key)
          logger.debug( "station: #{key} found")
          return message
        end
      end

      return nil
    end


    def expireStations
      logger.info( "expireStations")

      now = Time.now

      @mutex.synchronize do 
        # clean up station list 
        @stations.each{ |key, timestamp|
          if now - timestamp > @timeout
            @stations.delete( key)
          end
        }
      end

      return
    end

  end

end
