#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'


module APRS4R

  class WMRWeatherMessage < APRS4RBase

    @@WMR_HEADER = 0xFF

    @@WMR_MESSAGE_TYPE_WIND = 0x00
    @@WMR_MESSAGE_TYPE_RAIN = 0x01
    
    @@WMR_MESSAGE_TYPE_THERMO_HYGRO_INDOOR = 0x02
    @@WMR_MESSAGE_TYPE_THERMO_HYGRO_OUTDOOR = 0x03
    @@WMR_MESSAGE_TYPE_THERMO_INDOOR = 0x04
    @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 = 0x05
    @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2 = 0x06

    @@WMR_MESSAGE_TYPE_MINUTE = 0x0E
    @@WMR_MESSAGE_TYPE_DATE = 0x0F


    @@WMR_MESSAGE_LENGTH = { 
      @@WMR_MESSAGE_TYPE_WIND => 11, # wind sensor data
      @@WMR_MESSAGE_TYPE_RAIN => 16, # rain sensor data
      
      @@WMR_MESSAGE_TYPE_THERMO_HYGRO_INDOOR => 9,  # indoor thermo/hygro sensor
      @@WMR_MESSAGE_TYPE_THERMO_HYGRO_OUTDOOR => 9,  # outdoor thermo/hygro sensor
      @@WMR_MESSAGE_TYPE_THERMO_INDOOR => 7,  # indoor thermo sensor
      @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 => 13,  # indoor thermo/hygbaro sensor
      @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2 => 14,  # indoor thermo/hygbaro sensor
      
      @@WMR_MESSAGE_TYPE_MINUTE => 5, # clock information (minute)
      @@WMR_MESSAGE_TYPE_DATE => 9  # clock information (hour)
    }


    attr_reader :type, :payload, :checksum
    attr_writer :type, :payload, :checksum


    @logger = APRS4RLogger.get_logger( "WMRWeatherMessage")


    def initialize( type, payload, checksum)
      logger.info( "initialize( type, payload, checksum)")

      @type = type
      @payload = payload
      @checksum = checksum

      return
    end


    def is_valid?
      logger.info( "is_valid?")
      
      value = 0 
      value += @@WMR_HEADER 
      value += @@WMR_HEADER

      value += type.to_i
      
      payload.each{ |data|
        value += data.to_i
      }

      if checksum == (value & 0xFF)
        return true
      end
      
      return false
    end


    def has_wind?
      return type == @@WMR_MESSAGE_TYPE_WIND
    end

    def has_rain?
      return type == @@WMR_MESSAGE_TYPE_RAIN
    end

    def has_thermo?
      return type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_INDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_OUTDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_INDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 || 
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
    end

    def has_thermo_indoor?
      return type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_INDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_INDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 || 
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
    end

    def has_thermo_outdoor?
      return type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_OUTDOOR
    end

    def has_hygro?
      return type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_INDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_OUTDOOR ||
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 || 
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
    end

    def has_baro?
      return type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1 || 
        type == @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
    end

    def has_minute?
      return type == @@WMR_MESSAGE_TYPE_MINUTE || type == @@WMR_MESSAGE_TYPE_DATE
    end

    def has_date?
      return type == @@WMR_MESSAGE_TYPE_DATE
    end


    def wind_direction
      direction = 0.0

      if has_wind? && payload.length >= 3

        direction = low_nibble( payload[2]) * 100.0 + high_nibble( payload[1]) * 10.0 + low_nibble( payload[1])
      end

      return direction
    end


    def wind_avg
      speed = 0.0

      if has_wind? && payload.length >= 6
        speed = low_nibble( payload[5]) * 10.0 + high_nibble( payload[4]) + low_nibble( payload[4]) / 10.0

      end
      
      return speed
    end


    def wind_gust
      speed = 0.0

      if has_wind? && payload.length >= 4
        speed = high_nibble( payload[3]) * 10.0 + low_nibble( payload[3]) + high_nibble( payload[2]) / 10.0
        
      end

      return speed
    end


    def rain_1h
      rain = 0.0

      if has_rain? && payload.length >= 3
        rain = low_nibble( payload[2]) * 100.0 + high_nibble( payload[1]) * 10.0 + low_nibble( payload[1])

        # convert to m/hr
        rain /= 100.0
      end

      return rain
    end


    def rain_24h
      rain = 0.0

      if has_rain? && payload.length >= 5
        rain = high_nibble( payload[4]) * 1000.0 + low_nibble( payload[4]) * 100.0 + high_nibble( payload[3]) * 10.0 + low_nibble( payload[3]) + high_nibble( payload[2]) / 10.0

        # convert to m/hr
        rain /= 100.0
      end

      return rain
    end


    def rain_yesterday
      rain = 0.0

      if has_rain? && payload.length >= 7
        rain = high_nibble( payload[6]) * 1000.0 + low_nibble( payload[6]) * 100.0 + high_nibble( payload[5]) * 10.0 + low_nibble( payload[5])

        # convert to m/hr
        rain /= 100.0
      end

      return rain
    end


    def temperature
      temperature = 0.0

      if has_thermo? && payload.length >= 3
        temperature = high_nibble( payload[2] & 0x30) * 100.0 + low_nibble( payload[2]) * 10.0 + high_nibble( payload[1]) + low_nibble( payload[1]) / 10.0 

        if (payload[2].to_i & 0x80) == 0x80
          temperature *= -1.0
        end
      end

      return temperature
    end

    
    def humidity
      humidity = 0.0

      if has_hygro? && payload.length >= 4
        humidity = high_nibble( payload[3]) * 10.0 + low_nibble( payload[3])
      end

      return humidity
    end


    def dewpoint
      dewpoint = 0.0
      
      if has_baro? && payload.length >= 5
        dewpoint = high_nibble( payload[4]) * 10.0 + low_nibble( payload[4])
      end
      
      return dewpoint
    end


    def pressure_base
      pressure = 0

      if has_baro? 
        case type
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1
          if payload.length >= 6
            pressure = payload[5]
          end
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
          if payload.length >= 7
            pressure = ((payload[6] & 0x01) << 8) + payload[5]
          end
        end
      end

      return pressure
    end


    def pressure_absolute
      # absolute = base + fixed offset
      pressure = pressure_base

      if has_baro? 
        case type
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1
          pressure += 795.0
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
          pressure += 600.0
        end
      end

      return pressure
    end


    def pressure_relative
      # relative = base + sea level offset + correction (if neccessary)
      pressure = pressure_base

      if has_baro?
        case type
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR1
          if payload.length >= 9
            offset = high_nibble( payload[8]) * 100.0 + low_nibble( payload[8]) * 10.0 + high_nibble( payload[7]) + low_nibble( payload[7]) / 10.0
            offset += 1000.0 if offset < 400.0
            pressure += offset
          end
        when @@WMR_MESSAGE_TYPE_THERMO_HYGRO_BARO_INDOOR2
          if payload.length >= 10
            offset = high_nibble( payload[9]) * 1000.0 + low_nibble( payload[9]) * 100.0 + high_nibble( payload[8]) * 10.0 + low_nibble( payload[8]) + high_nibble( payload[7]) / 10.0
            pressure += offset
          end
        end
      end
      
      return pressure
    end


    def minute
      minute = 0

      if has_minute? && payload.length >= 1 
        minute = high_nibble( payload[0]) * 10 + low_nibble( payload[0])
      end

      return minute
    end

    def hour
      hour = 0

      if has_date? && payload.length >= 2
        hour = high_nibble( payload[1]) * 10 + low_nibble( payload[1])
      end

      return hour
    end

    def day
      day = 0

      if has_date? && payload.length >= 3
        day = high_nibble( payload[2]) * 10 + low_nibble( payload[2])
      end

      return day
    end

    def month 
      month = 0
      
      if has_date? && payload.length >= 4
        month = high_nibble( payload[3]) * 10 + high_nibble( payload[3])
      end
      
      return month
    end

    def year 
      year = 0
      
      if has_date? && payload.length >= 5
        year = 2000 + high_nibble( payload[4]) * 10 + low_nibble( payload[4])
      end

      return year
    end
    

    def low_nibble( value)
      return value & 0x0F
    end

    def high_nibble( value)
      return (value & 0xF0) >> 4
    end

    def WMRWeatherMessage.WMR_HEADER
      return @@WMR_HEADER
    end

    def WMRWeatherMessage.message_length( type)
      return @@WMR_MESSAGE_LENGTH[type]
    end

  end

end
