#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/MICEMessage'

require 'aprs4r/ConfigurationAttribute'


module APRS4R 

  class APRSMessage < APRS4RBase

    attr_reader :name, :type, :source, :destination, :path, :payload
    attr_writer :name, :type, :source, :destination, :path, :payload


    @@APRS_PATH_RELAY = "RELAY"
    @@APRS_PATH_WIDE  = "WIDE"
    @@APRS_PATH_TRACE = "TRACE"

    @@APRS_POSITION = { 
      # common messages
      '!' => true, '=' => true, '/' => true, '@' => true, ';' => true,
      # mic-e messages
      '`' => true, '\'' => true, 0x1c => true, 0x1d => true
    }

    @@APRS_POSITION_LATITUDE_INDEX =  { '!' =>  1, '=' =>  1, '/' =>  8, '@' =>  8, ';' => 18}
    @@APRS_POSITION_LONGITUDE_INDEX = { '!' => 10, '=' => 10, '/' => 17, '@' => 17, ';' => 27}

    @@APRS_SYMBOL_TABLE_INDEX = { 
      # common messages
      '!' => 9, '=' => 9, '/' => 16, '@' => 16, ';' => 26,
      # mic-e messages
      '`' => 8, '\'' => 8, 0x1c => 8, 0x1d => 8
    }

    @@APRS_SYMBOL_CODE_INDEX = { 
      # common messages
      '!' => 19, '=' => 19, '/' => 26, '@' => 26, ';' => 36,
      # mic-e messages
      '`' => 7, '\'' => 7, 0x1c => 7, 0x1d => 7
    }

    @@APRS_COMPRESSED_POSITION = { 
      '!' => true, '=' => true, '/' => true, '@' => true
    }

    @@APRS_COMPRESSED_POSITION_LATITUDE_INDEX = { '!' => 2, '=' => 2, '/' => 9, '@' => 9}
    @@APRS_COMPRESSED_POSITION_LONGITUDE_INDEX = { '!' => 6, '=' => 6, '/' => 13, '@' => 13}

    @@APRS_COMPRESSED_SYMBOL_TABLE_INDEX = {
      '!' => 1, '=' => 1, '/' => 8, '@' => 8
    }

    @@APRS_COMPRESSED_SYMBOL_CODE_INDEX = { 
      '!' => 10, '=' => 10, '/' => 17, '@' => 17
    }


    @@APRS_EXTENSION_INDEX = { '!' => 20, '=' => 20, '/' => 27, '@' => 27 }
    @@APRS_EXTENSION_LENGTH = 7

    @@APRS_WEATHER_INDEX = { '_' => 10, '!' => 20, '=' => 20, '/' => 27, '@' => 27 } 
    @@APRS_WEATHER_LENGTH = 0

    @@APRS_COMMENT_INDEX = { '!' => 20, '=' => 20, '/' => 27, '@' => 27, ';' => 37 }

    @@APRS_SYMBOL_WEATHER = '_'

    @@APRS_SYMBOL_CODE_NAME = { 
      '!' => "0", 
      '\"' => "1", '#' => "2",  '$' => "3",  '%' => "4",  '&' => "5",
      '\'' => "6", '(' => "7",  ')' => "8",  '*' => "9",  '+' => "10", 
      ',' => "11", '-' => "12", '.' => "13", '/' => "14", '0' => "15",
      '1' => "16", '2' => "17", '3' => "18", '4' => "19", '5' => "20",
      '6' => "21", '7' => "22", '8' => "23", '9' => "24", ':' => "25",
      ';' => "26", '<' => "27", '=' => "28", '>' => "29", '?' => "30",
      '@' => "31", 'A' => "32", 'B' => "33", 'C' => "34", 'D' => "35",
      'E' => "36", 'F' => "37", 'G' => "38", 'H' => "39", 'I' => "40",
      'J' => "41", 'K' => "42", 'L' => "43", 'M' => "44", 'N' => "45",
      'O' => "46", 'P' => "47", 'Q' => "48", 'R' => "49", 'S' => "50",
      'T' => "51", 'U' => "52", 'V' => "53", 'W' => "54", 'X' => "55",
      'Y' => "56", 'Z' => "57", '[' => "58", '\\' => "59", ']' => "60",
      '^' => "61", '_' => "62", '`' => "63", 'a' => "64", 'b' => "65",
      'c' => "66", 'd' => "67", 'e' => "68", 'f' => "69", 'g' => "70",
      'h' => "71", 'i' => "72", 'j' => "73", 'k' => "74", 'l' => "75",
      'm' => "76", 'n' => "77", 'o' => "78", 'p' => "79", 'q' => "80",
      'r' => "81", 's' => "82", 't' => "83", 'u' => "84", 'v' => "85",
      'w' => "86", 'x' => "87", 'y' => "88", 'z' => "89", '{' => "90",
      '|' => "91", '}' => "92", '~' => "93"
      
    }

    @@APRS_PHG_POWER = [0, 1, 4, 9, 16, 25, 36, 49, 64, 81]
    @@APRS_PHG_HEIGHT = [10, 20, 40, 80, 160, 320, 640, 1280, 2560, 5120]
    @@APRS_PHG_GAIN = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
    @@APRS_PHG_DIRECTION = ["omni", "45/NE", "90/E", "135/SE", "180/S", "225/SW", "270/W", "315/NW", "360/N"]
    
    
    @logger = APRS4RLogger.get_logger( "APRSMessage")


    def initialize( source = APRSCall.new, destination = APRSCall.new, path = Array.new, payload = nil)
      logger.info( "initialize( source, destination, path, payload)")
      
      @name = "message"
      @type = "APRSMessage"

      @source = source
      @destination = destination
      @path = path
      @payload = payload

      @type = nil
      
      return
    end


    def clone
      logger.info( "clone")
      
      message = APRSMessage.new

      message.source = @source.clone
      message.destination = @destination.clone

      path = Array.new
      @path.each{ |entry| path << entry.clone} if @path
      message.path = path

      message.payload = String.new( @payload)

      return message
    end


    def duplicate_key
      logger.info( "duplicate_key")
      
      key = String.new

      # ignore path for duplicate detection
      key << source.to_s << ">" << destination.to_s
      key << ":" << payload

      return key
    end


    def to_yaml_properties
      %w{@source @destination @path @payload}
    end

    
    def attributes

      attributes = [
                    ConfigurationAttribute.new( "name",        "text",  false,  10, nil,   "Name"), 
                    ConfigurationAttribute.new( "type",        "text",  false, 10, nil,   "Typ"), 
                    ConfigurationAttribute.new( "source",      "text",  true, 10, "MYCALL", "Rufzeichen"),
                    ConfigurationAttribute.new( "destination", "text",  true, 10, "AP4R09", "Ziel"),
                    ConfigurationAttribute.new( "path",        "array", true, 20, ["WIDE1-1", "WIDE2-1"], "Pfad"),
                    ConfigurationAttribute.new( "payload",     "text",  true, 40, "=4900.00NI00825.00E&Text", "Payload", "/aprs4r-web/detail.html")
                   ]

      return attributes
    end


    def has_path?
      logger.info( "has_path?")

      if @path && !@path.empty?
        return true
      end

      return false
    end


    def is_local?
      logger.info( "is_local?")

      if !has_path?
        return true
      end

      @path.each{ |entry|
        call = APRSCall.parse( entry)

        return false if call.repeated?
      }

      return true
    end


    def type
      type = nil

      if payload.length > 0 
        type = payload[0].chr
      end

      return type
    end


    def has_symbol?
      logger.info( "has_symbol?")

      return @@APRS_SYMBOL_TABLE_INDEX.has_key?( type)
    end


    def symbol_table
      logger.info( "symbol_table")

      if payload.length > 0 
        index = nil

        if is_compressed?
          index = @@APRS_COMPRESSED_SYMBOL_TABLE_INDEX[type]
        else
          index = @@APRS_SYMBOL_TABLE_INDEX[type]
        end

        if index && index < payload.length
          return payload[index].chr
        end
      end

      return nil
    end


    def symbol_code
      logger.info( "symbol_code")

      if payload.length > 0 
        index = nil

        if is_compressed?
          index = @@APRS_COMPRESSED_SYMBOL_CODE_INDEX[type]
        else
          index = @@APRS_SYMBOL_CODE_INDEX[type]
        end

        if index && index < payload.length
          return payload[index].chr
        end
      end
      
      return nil
    end

    
    def symbol_name
      logger.info( "symbol_name")

      code = symbol_code
      name = @@APRS_SYMBOL_CODE_NAME[code]

      return name
    end


    def is_compressed?

      # exclude all position less messages
      if ! @@APRS_COMPRESSED_POSITION.has_key?( type)
        return false
      end
      
      # check for uncompressed position 
      if ! has_position? 
        return true
      end

      return false
    end


    def has_position?
      logger.info( "has_position?")

      if ! @@APRS_POSITION.has_key?( type) 
        return false
      end

      # common messages
      position_match = /[\d\ ]{4}\.[\d\ ]{2}[NS].[\d\ ]{5}\.[\d\ ]{2}[EW]/

      return true if payload =~ position_match

      # mic-e messages
      return true if is_mice?

      return false
    end
    

    def latitude
      logger.info( "latitude")

      latitude = 0.0

      if has_position?

        if is_mice?
          latitude = MICEMessage.latitude( self)
        else
          index = @@APRS_POSITION_LATITUDE_INDEX[type]
          length = 8
          
          if index + length < payload.length
            latitude = APRSMessage.degree2decimal( payload[index...index+length])
          end
        end
      end

      return latitude
    end


    def latitude=( latitude)
      logger.info( "latitude=( latitude)")

      if has_position?
        if is_mice?
          # TODO: latitude= for MICEMessage
        else
          index = @@APRS_POSITION_LATITUDE_INDEX[type]
          length = 8

          if index + length < payload.length
            payload[index...index+length] = APRSMessage.decimal2latitude( latitude)
          end
        end
      end

      return
    end


    def longitude
      logger.info( "longitude")

      longitude = 0.0

      if has_position?

        if is_mice?
          longitude = MICEMessage.longitude( self)
        else
          index = @@APRS_POSITION_LONGITUDE_INDEX[type]
          length = 9

          if index + length < payload.length
            longitude = APRSMessage.degree2decimal( payload[index...index+length])
          end
        end
      end

      return longitude
    end

    
    def longitude=( longitude)
      logger.info( "longitude=( longitude)")

      if has_position?
        if is_mice?
          # TODO: longitude= for MICEMessage
        else
          index = @@APRS_POSITION_LONGITUDE_INDEX[type]
          length = 8

          if index + length < payload.length
            payload[index...index+length] = APRSMessage.decimal2longitude( longitude)
          end
        end
      end

      return
    end


    def has_extension?
      logger.info( "has_extension?")

      course = has_course?
      logger.info( "course: #{course}")

      phg = has_phg?
      logger.info( "phg: #{phg}")

      return course || phg
    end


    def has_comment?
      logger.info( "has_comment?")

      if has_weather? 
        return false
      end

      return @@APRS_COMMENT_INDEX.has_key?( type)
    end


    def comment
      logger.info( "comment")
      
      comment = String.new
      index = @@APRS_COMMENT_INDEX[type]

      if has_extension?
        index += @@APRS_EXTENSION_LENGTH
      end

      if has_comment? && payload.length > index
        comment = payload[index...payload.length]
      end

      return comment
    end


    def has_status?
      logger.info( "has_status?")

      return type == '>' || is_mice?
    end


    def status
      logger.info( "status")

      status = String.new

      if has_status?

        if is_mice?
          status = payload[9...payload.length] if payload.length > 9
        else
          status = payload[1...payload.length] if payload.length > 1
        end
      end
      
      return status
    end

    
    def has_weather?
      # NOTE: compare type id with symbol (equal for weather stations)

      if type != @@APRS_SYMBOL_WEATHER && symbol_code != @@APRS_SYMBOL_WEATHER
        return false
      end

      return type == @@APRS_SYMBOL_WEATHER || symbol_code == @@APRS_SYMBOL_WEATHER
    end

    
    def weather_wind_direction
      direction = 0
      index = @@APRS_WEATHER_INDEX[type]

      if index && has_weather? && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
        if data && data.length > 0 
          direction = data[0][0..2].to_i
        end
        # direction = payload[index..index+3].to_i
      end

      return direction
    end


    def weather_wind_speed
      speed = 0
      index = @@APRS_WEATHER_INDEX[type]

      if index && has_weather? && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
        if data && data.length > 0 
          speed = data[0][4..6].to_i
        end
        # speed = payload[index+4..index+6].to_i
      end

      return speed
    end


    def weather_wind_gust
      speed = 0

      index = @@APRS_WEATHER_INDEX[type]

      if has_weather? && index && payload.lengt >= index
        data = payload[index..payload.length].to_s.scan( /g\d\d\d/)
        if data && data.length > 0 
          speed = data[0][1..3].to_i
        end
      end

      return speed
    end


    def weather_temperature
      temperature = 0

      index = @@APRS_WEATHER_INDEX[type]

      if has_weather? && index && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /t\d\d\d/)
        if data && data.length > 0 
          temperature = data[0][1..3].to_i
        end
        # temperature = payload[index+12..index+14].to_i
      end

      return temperature
    end


    def weather_rain
      rain = 0 

      index = @@APRS_WEATHER_INDEX[type]

      if has_weather? && index && payload.length >= index 
        data = payload[index..payload.length].to_s.scan( /r\d\d\d/)
        if data && data.length > 0 
          rain = data[0][1..3].to_i
        end
      end

      return rain
    end


    def weather_humidity
      humidity = 0

      index = @@APRS_WEATHER_INDEX[type]

      if has_weather? && index && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /h\d\d/)
        if data && data.length > 0 
          humidity = data[0][1..2].to_i
        end
      end

      return humidity
    end


    def weather_pressure
      pressure = 0 
      
      index = @@APRS_WEATHER_INDEX[type]

      if has_weather? && index && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /b\d\d\d\d\d/)
        if data && data.length > 0
          pressure = data[0][1..5].to_f / 10.0
        end
      end

      return pressure
    end


    def has_course?
      logger.info( "has_course?")

      index = @@APRS_COMMENT_INDEX[type]

      if symbol_code == @@APRS_SYMBOL_WEATHER
        return false
      end

      if index && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
        if data && data.length > 0 
          return true 
        end
      end

      return true if is_mice?

      return false
    end


    def course_direction
      direction = 0

      if has_course?

        if is_mice?
          direction = MICEMessage.course_direction( self)
        else
          index = @@APRS_COMMENT_INDEX[type]

          if index && has_course? && payload.length >= index
            data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
            if data && data.length > 0 
              direction = data[0][0..2].to_i
            end
            # direction = payload[index..index+3].to_i
          end
        end

      end

      return direction
    end


    def course_speed
      speed = 0

      if has_course?

        if is_mice?
          speed = MICEMessage.course_speed( self)
        else
          index = @@APRS_COMMENT_INDEX[type]

          if index && has_course? && payload.length >= index
            data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
            if data && data.length > 0 
              speed = data[0][4..6].to_i
            end
            # speed = payload[index+4..index+6].to_i
          end
        end

      end

      return speed
    end


    def has_phg?
      logger.info( "has_phg?")

      index = @@APRS_COMMENT_INDEX[type]

      if index && payload.length >= index
        data = payload[index..payload.length].to_s.scan( /PHG\d\d\d\d/)
        if data && data.length > 0 
          return true 
        end
      end

      return false
    end


    def phg_power
      power = 0
      index = @@APRS_COMMENT_INDEX[type]

      if index && has_phg? && payload.length >= (index+7)
        value = payload[index+3].chr.to_i

        if value >= 0 && value < 10 
          power = @@APRS_PHG_POWER[value]
        end
      end

      return power
    end


    def phg_height
      height = 0
      index = @@APRS_COMMENT_INDEX[type]

      if index && has_phg? && payload.length >= (index+7)
        value = payload[index+4].chr.to_i

        if value >= 0 && value < 10 
          height = @@APRS_PHG_HEIGHT[value]
        end
      end

      return height
    end


    def phg_gain
      gain = 0
      index = @@APRS_COMMENT_INDEX[type]

      if index && has_phg? && payload.length >= (index+7)
        value = payload[index+5].chr.to_i

        if value >= 0 && value < 10 
          gain = @@APRS_PHG_GAIN[value]
        end
      end

      return gain
    end


    def phg_direction
      direction = 0
      index = @@APRS_COMMENT_INDEX[type]

      if index && has_phg? && payload.length >= (index+7)
        value = payload[index+6].chr.to_i

        if value >= 0 && value < 10 
          direction = @@APRS_PHG_DIRECTION[value]
        end
      end

      return direction
    end

    
    def is_object?
      return type == ';'
    end

    
    def object_address
      address = "" 

      if is_object? && payload.length >= 11
        address = payload[1..9].strip
      end
        
      return address
    end


    def is_message?
      return type == ':'
    end


    def message_recipient
      recipient = ""

      if is_message? && payload.length >= 11
        values = payload[1..10].split( /:/)

        recipient = values[0] if values.length >= 1
      end
        
      return recipient.strip
    end

    
    def is_query?
      return type == '?'
    end


    def is_mice?
      return type == '`' || type == '\'' || type == 0x1c || type == 0x1d
    end
    


    def to_s
      buffer = String.new

      source = "unknown"
      source = @source if @source
      destination = "unknown"
      destination = @destination if @destination
      path = ["EMTPY"]
      path = @path if @path
      payload = "unknown"
      payload = @payload if @payload

      # buffer << "APRSMessage: "
      buffer << source.to_s << " -> "
      buffer << destination.to_s << " via "
      
      value = String.new
      for i in 0...path.length

        if i != 0 
          value += ", "
        end

        value += path[i].to_s
      end

      buffer << "[" << value << "]: "
      buffer << "(" << payload << ")"

      return buffer.to_s
    end


    def APRSMessage.degree2decimal( position)
      logger.info( "degree2decimal")

      value = 0.0

      if position.nil? || position.length < 8
        return value
      end

      case position[position.length-1].chr
      when 'N'
        value = position[0...2].to_f + (position[2...4].to_f + position[5...7].to_f / 100.0) / 60.0
        
      when 'S'
        value = position[0...2].to_f + (position[2...4].to_f + position[5...7].to_f / 100.0) / 60.0
        value *= -1.0

      when 'E'
        value = position[0...3].to_f + (position[3...5].to_f + position[6...8].to_f / 100.0) / 60.0

      when 'W'
        value = position[0...3].to_f + (position[3...5].to_f + position[6...8].to_f / 100.0) / 60.0
        value *= -1.0

      end

      return value
    end


    def APRSMessage.latitude2decimal( latitude)
      logger.info( "latitude2decimal")

      value = 0.0

      if latitude.nil? || latitude.length < 8
        return value
      end

      latitude_match = /^[\d\ ]{4}\.[\d\ ]{2}[NS]$/

      if latitude =~ latitude_match
        value = latitude[0...2].to_f + (latitude[2...4].to_f + latitude[5...7].to_f / 100.0) / 60.0
        value *= -1.0 if latitude[7] == 'S'
      end

      return value
    end


    def APRSMessage.longitude2decimal( longitude)
      logger.info( "longitude2decimal")

      value = 0.0

      if longitude.nil? || longitude.length < 9
        return value
      end

      longitude_match = /^[\d\ ]{5}\.[\d\ ]{2}[EW]$/

      if longitude =~ longitude_match
        value = longitude[0...3].to_f + (longitude[3...5].to_f + longitude[6...8].to_f / 100.0) / 60.0
        value *= -1.0 if longitude[8] == 'W'
      end

      return value
    end


    def APRSMessage.decimal2latitude( latitude)
      logger.info( "decimal2latitude( latitude)")

      degrees = latitude.abs.floor
      print "degrees: #{degrees}\n"
      minutes = (latitude.abs - degrees) * 60.0
      print "minutes: #{minutes}\n"

      orientation = 'N'
      orientation = 'S' if latitude < 0.0
      
      value = sprintf( "%02.2i%02.2f%s", degrees, minutes, orientation)

      return value
    end


    def APRSMessage.decimal2longitude( longitude)
      logger.info( "decimal2longitude( longitude)")

      degrees = longitude.abs.floor
      print "degrees: #{degrees}\n"
      minutes = (longitude.abs - degrees) * 60.0
      print "minutes: #{minutes}\n"

      orientation = 'E'
      orientation = 'W' if longitude < 0.0
      
      value = sprintf( "%03.3i%02.2f%s", degrees, minutes, orientation)

      return value
    end


    def APRSMessage.APRS_PATH_RELAY
      return @@APRS_PATH_RELAY
    end

    def APRSMessage.APRS_PATH_WIDE
      return @@APRS_PATH_WIDE
    end

    def APRSMessage.APRS_PATH_TRACE
      return @@APRS_PATH_TRACE
    end

    def APRSMessage.attributes 
      
      attributes = [
                    ConfigurationAttribute.new( "source", 01, "text", "Quelle", 10),
                    ConfigurationAttribute.new( "destination", 02, "text", "Ziel", 10), 
                    ConfigurationAttribute.new( "path", 03, "text", "Pfad", 10),
                    ConfigurationAttribute.new( "payload", 04, "text", "Nutzdaten", 40)
                   ]

      return attributes
    end

  end

end
