#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'io/nonblock'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSMessage'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/WMRWeatherMessage'
require 'aprs4r/WMRWeatherPluginConfiguration'


module APRS4R

  class WMRWeatherPlugin < WeatherPlugin

    @logger = APRS4RLogger.get_logger( "WMRWeatherPlugin")

    
    def initialize( configuration)
      logger.info( "initialize( configuration)")

      super( configuration)

      @file = configuration.file

      @port = SerialPort.new( @file, 9600, 8, 1, SerialPort::NONE)
      @port.nonblock = true 
      @port.sync = true
      @port.flow_control = SerialPort::HARD

      Thread.new{ runWMRThread }

      return
    end


    def start
      logger.info( "start")

      Thread.new { 
        runWMRThread
      }

      return
    end


    def runWMRThread
      logger.info( "runWMRThread")
      
      while true 
        readWeatherData
      end

      return
    end


    def readWeatherData
      logger.info( "readWeatherData")

      record = readRecord

      if record 
        if record.has_wind?
          logger.info( "wind: direction: #{record.wind_direction}, avg: #{record.wind_avg}, gust: #{record.wind_gust}")
          @wind_direction = record.wind_direction
          @wind_avg = record.wind_avg
          @wind_gust = record.wind
        end
        
        if record.has_rain?
          logger.info( "rain: 1h: #{record.rain_1h}, 24h: #{record.rain_24h}, yesterday: #{record.rain_yesterday}")
          @rain_1h = record.rain_1h
          @rain_total = record.rain_24h
          @rain_yesterday = record.rain_yesterday
        end
        
        if record.has_thermo_outdoor?
          logger.info( "thermo: temperature: #{record.temperature}, humidity: #{record.humidity}, dewpoint: #{record.dewpoint}")
          @temperature = record.temperature
          @humidity = record.humidity
          @dewpoint = record.dewpoint
        end

        if record.has_baro?
          logger.info( "baro: pressure: #{record.pressure_relative}")
          @pressure = record.pressure_relative
        end
      else 
        logger.warn( "empty record received")
      end
      
      return
    end


    def readRecord
      logger.info( "readRecord")

      type = 0x10
      payload = String.new
      checksum = 0

      # read until HEADER
      value = @port.getc
      while value != WMRWeatherMessage.WMR_HEADER
        logger.debug( "value: #{value.chr.to_i}")
        value = @port.getc
      end

      # read all HEADER
      while value == WMRWeatherMessage.WMR_HEADER
        value = @port.getc
      end

      type = value 
      logger.debug( "type: #{type}")

      length = WMRWeatherMessage.message_length( type)
      logger.debug( "length: #{length}")
      
      payload = Array.new
      if length != nil && length > 4
        (length-4).times { 
          value = @port.getc
          logger.debug( "payload byte: #{value}")
          payload << value
        }
      end

      checksum = @port.getc
      logger.debug( "checksum: #{checksum}")

      # create wmr weather message
      record = WMRWeatherMessage.new( type, payload, checksum)
      
      if record.is_valid? 
        logger.debug( "message is valid")
        return record
      end
      
      logger.warn( "bad checksum, returning nil")

      return nil
    end

  end

end
