#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Michael Conrad <do5mc@friggleware.de> and
#                        Andreas Bier <dl1hrc@web.de>
#                        Christopher A. Kantarjiev <cak@dimebank.com>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'serialport'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'aprs4r/APRS4RBase'
require 'aprs4r/APRS4RLogger'


module APRS4R 

  class KISSDevice < APRS4RBase

    @@FEND = 0xC0
    @@FESC = 0xDB
    @@TFEND = 0xDC
    @@TFESC = 0xDD

    @logger = APRS4RLogger.get_logger( "KISSDevice")
    

    def initialize( device, baudrate, call, speed, timeout, mode)
      logger.info( "initialize( device, baudrate, call, speed, timeout, mode)")
      
      @device = device
      @call = call

      @baudrate = 9600 
      @baudrate = baudrate if baudrate 

      @speed = 1200 
      @speed = speed if speed 

      @timeout = timeout
      @mode = mode

      @mutex = Mutex.new
      
      initKISSDevice()

      return
    end


    def initKISSDevice()
      logger.info( "initKISSDevice")

      begin 

        @mutex.synchronize do
          
          if @port 
            @port.close
          end
          
          @port = SerialPort.new( @device, @baudrate, 8, 1, SerialPort::NONE)
          @port.nonblock = true 
          @port.sync = true
          
          @port.flow_control = SerialPort::SOFT
          
          mode = @mode.downcase if @mode
          
          if mode.index( "tnc2/northlink") != nil
            initTNC2Northlink
          elsif mode.index( "tnc2/tapr") != nil
            initTNC2TAPR
          elsif mode.index( "tnc2/multi") != nil
            initTNC2Multi
          elsif mode.index( "tnc7") != nil
            initTNC7
          elsif mode.index( "tncx") != nil
            initTNCX
          elsif mode.index( "kenwood") != nil
            initKenwood
          elsif mode.index( "scs/ptc") != nil
            initSCSPTC
          elsif mode.index( "scs/tracker") != nil
            initSCSTracker
          elsif mode.index( "ot2") != nil
            initOpenTracker2
          else
            initDefault
          end
          
        end

      rescue Exception => ex
        logger.error( "initKISSDevice: ex: #{ex}")
      end

      return
    end


    def reinitKISSDevice()
      logger.info( "reinitKISSDevice")

      sleep 5

      initKISSDevice()

      return
    end


    def initTNC2Northlink
      logger.info( "initTNC2Northlink")

      # send exit frame 
      data = String.new
      
      #    data << @@FEND
      #    data << 0xFF
      #    data << @@FEND
      #    data << "\r" 

      #    logger.debug( "send KISS exit frame");
      #    @port.write( data)
      #    sleep 1

      data = String.new
      data << ?\C-X
      data << ?\C-[
      data << "@K"
      data << "\r"

      logger.debug( "send KISS init sequence")
      @port.write( data)
      sleep 1

      return
    end


    def initTNC2TAPR
      logger.info( "initTNC2TAPR")
      
      # send exit frame 
      data = String.new
      
      data << @@FEND
      data << 0xFF
      data << @@FEND
      data << "\r" 

      logger.debug( "send KISS exit frame");
      @port.write( data)
      sleep 1
      
      # enable kiss mode
      # logger.debug( "send: mycall #{@call}");
      # @port.write( "mycall #{@call}\r")
      # sleep 1

      # logger.debug( "send: hbaud #{@speed}");
      # @port.write( "hbaud #{@speed}\r");
      # sleep 1

      logger.debug( "send: kiss on ");
      @port.write( "kiss on\r");
      sleep 1

      logger.debug( "send: restart");
      @port.write( "restart\r")
      sleep 1

      return
    end


    def initTNC2Multi
      logger.info( "initTNC2Multi")
      
      # send exit frame 
      data = String.new 
      data << @@FEND
      data << 0xFF
      data << @@FEND
      data << 0x0D.chr

      logger.debug( "send KISS exit frame");
      @port.write( data)

      sleep 2    # WICHTIG, mit sleep 1 ging es nur jedes zweite Mal!!
      
      # enable kiss mode
      logger.debug( "send: mycall #{@call}");
      @port.write( "mycall #{@call}\r")
      sleep 1

      kiss = String.new
      kiss << 0x1B
      kiss << "@K"
      kiss << 0x0D

      logger.debug( "send: kiss on ");
      @port.write( kiss );
      sleep 1
      return
    end
    

    def initTNC7
      logger.info( "initTNC7")
      
      # nothing to do ;-) 

      return
    end


    def initTNCX
      logger.info( "initTNCX")
      
      # nothing to do ;-) 

      return
    end


    def initSCSPTC
      logger.info( "initSCSPTC")
      
      # send exit frame 
      data = String.new
      data << @@FEND
      data << 0xFF
      data << @@FEND
      data << "\r\n" 

      #    logger.debug( "send KISS exit frame");
      #    @port.write( data)
      #    sleep 5

      data = String.new
      data << 0x1B.chr
      @port.write( data)
      sleep 1

      # enable packet mode
      logger.debug( "send PACKET command")
      @port.write( "PACKET\r\n")
      sleep 1

      logger.debug( "send: baud #{@speed}");
      @port.write( "baud #{@speed}\r\n");
      sleep 1

      # enable kiss mode
      logger.debug( "send KISS command")
      @port.write( "KISS\r\n")
      sleep 1

      return
    end


    def initSCSTracker
      logger.info( "initSCSTracker")

      initTNC2Northlink

      return
    end


    def initOpenTracker2
      logger.info( "initOpenTracker2")
      
      logger.debug( "restart OT2") 
      @port.write( "\003\003\003\r\003\003\003\r") 
      sleep 2

      logger.debug( "send: AMODE KISS") 
      @port.write( "AMODE KISS\r") 
      sleep 2

      return
    end


    def initKenwood
      logger.info( "initKenwood")

      logger.debug( "restart kenwood tnc")
      @port.write( "TC 1\r")
      sleep 2
      @port.write( "TC 0\r")
      sleep 2

      # enable software flow control
      logger.debug( "send: TXFLOW ON")
      @port.write( "TXFLOW ON\r")
      sleep 1

      # enable software dcd
      logger.debug( "send: SOFTDCD ON")
      @port.write( "SOFTDCD ON\r")
      sleep 1

      # enable kiss mode
      logger.debug( "send: mycall #{@call}");
      @port.write( "mycall #{@call}\r")
      sleep 1
      
      logger.debug( "send: hbaud #{@speed}");
      @port.write( "hbaud #{@speed}\r");
      sleep 1
      
      logger.debug( "send: kiss on ");
      @port.write( "kiss on\r");
      sleep 1
      
      logger.debug( "send: restart");
      @port.write( "restart\r")
      sleep 1
      
      return
    end


    def initDefault
      logger.info( "initDefault")
      
      # send exit frame 
      data = String.new
      
      data << @@FEND
      data << 0xFF
      data << @@FEND
      data << 0x0D.chr

      logger.debug( "send KISS exit frame");
      @port.write( data)
      sleep 1
      
      # enable kiss mode
      # logger.debug( "send: mycall #{@call}");
      # @port.write( "mycall #{@call}\r")
      # sleep 1

      # logger.debug( "send: hbaud #{@speed}");
      # @port.write( "hbaud #{@speed}\r");
      # sleep 1

      logger.debug( "send: kiss on ");
      @port.write( "kiss on\r");
      sleep 1

      logger.debug( "send: restart");
      @port.write( "restart\r")
      sleep 1

      return
    end


    def readFrame
      logger.info( "readFrame")

      frame = nil

      begin 
        Timeout::timeout( @timeout) do 
          # read until FEND 
          value = @port.getc
          while value != @@FEND 
            
            raise Exception.new( "Error reading from serial port") if value.nil?
            
            logger.info( "read wire byte: #{value} #{value.chr}")
            
            value = @port.getc
          end
          
          # read all FENDs
          while value == @@FEND
            value = @port.getc

            raise Exception.new( "Error reading from serial port") if value.nil?
          end

          # read command byte
          command = value

          if (command & 0x0F) != 0 
            logger.warn( "Error: reading invalid command byte from tnc: #{command}")
            return nil
          end
          
          # read frame
          frame = String.new
          value = @port.getc
          while value != @@FEND
            frame << value
            value = @port.getc
            
            raise Exception.new( "Error reading from serial port") if value.nil?
          end

          # drop crc bytes if smack extension 
          if (command & 0xF0) == 0x80
            if frame.length >= 2 
              frame = frame[0...frame.length-2]
            end
          end
        end
      rescue Exception => ex
        logger.warn( "readFrame: ex: #{ex}")
        reinitKISSDevice()
        return nil
      end

      return decodeKISS( frame)
    end


    def writeFrame( message)
      logger.info( "writeFrame( message)")

      begin

        data = String.new
        
        data << @@FEND

        # we only support the first port and make no smack
        data << 0x00
        data << encodeKISS( message)
        data << @@FEND

        @mutex.synchronize do
        
          # if tnc is an opentracker 2, wait 1/4 second
          if @mode.downcase == "kiss/ot2"
            sleep 0.250
          end
          
          @port.puts( data)
        end

      rescue Exception => ex
        logger.warn( "writeFrame::ex: " + ex)
      end
      
      return
    end


    def encodeKISS( payload)
      logger.info( "encodeKISS( payload)")

      data = String.new

      for i in 0...payload.length

        value = payload[i]

        # do kiss encoding (FEND->FESC+TFEND, FESC->FESC+TFESC)
        if value == @@FEND
          data << @@FESC << @@TFEND
        elsif value == @@FESC
          data << @@FESC << @@TFESC
        else 
          data << value
        end

      end

      return data    
    end


    def decodeKISS( payload)
      logger.info( "decodeKISS( payload)")

      if payload.nil?
        logger.warn( "nil payload field")
        return nil
      end

      data = String.new

      for i in 0...payload.length

        value = payload[i]

        # do kiss decoding (FESC+TFEND->FEND, FESC+TFESC->FESC)
        if value == @@FESC
          value = payload[i+=1]
          
          if value == @@TFEND
            data << @@FEND
          elsif value == @@TFESC
            data << @@FESC
          end
        else 
          data << value
        end 

      end

      return data
    end
    
  end

end
