#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2007 by Michael Conrad <do5mc@friggleware.de> and
#                        Andreas Bier <dl1hrc@web.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'socket'
require 'thread'
require 'timeout'
require 'io/nonblock'

require 'aprs4r/APRS4RLogger'

require 'aprs4r/APRSCall'
require 'aprs4r/Socket'


module APRS4R

  class ISSocket < Socket

    @logger = APRS4RLogger.get_logger( "ISSocket")
    
    
    def initialize( configuration, socket = nil)
      logger.info( "initialize( configuration | socket)")
      
      super( configuration)
      
      logger.debug( "configuration: #{configuration}")
      
      if configuration 
        # socket client mode 
        @clientMode = true 
        
        @hosts = configuration.hosts
        @port = configuration.port
        @username = configuration.username
        @password = ISSocket.generatePassword( @username)
        @filter = configuration.filter
        @timeout = configuration.timeout
      else 
        # socket server mode 
        @enable = true
        @clientMode = false
        @timeout = 0
        
        @socket = socket
        @socket.nonblock = true
        @socket.sync = true 
      end
      
      logger.debug( "clientMode: #{@clientMode}")
      
      # if @enable 
      #   initISSocket()
      # end
      
      return
    end
    
    
    def initSocket()
      logger.info( "initSocket()")
      
      if ! @enable 
        return
      end
      
      if @clientMode
        initClientSocket
      else 
        initServerSocket
      end
      
      return
    end
    
    
    def initClientSocket()
      logger.info( "initClientSocket()")
      
      begin 
        
        host = @hosts.first
        
        logger.debug( "connecting to #{host}:#{@port}")
        @socket = TCPSocket.new( host, @port)
        
        @socket.nonblock = true
        @socket.sync = true 
        
        # send login sequence
        username = "user " + @username
        password = "pass " + @password
        software = "vers aprs4r #{APRS4R::APRS4RVersion}"
        
        @socket.puts( username + " " + password + " " + software)
        @socket.puts( @filter)
      rescue Exception => ex
        logger.warn( "initClientSocket::ex: #{ex}")
        
        # wait 5 seconds before next try
        sleep 5
        
        # rotate aprs server list 
        host = @hosts.delete_at( 0)
        @hosts << host
        
        retry
      end
      
      return
    end
    
    
    def initServerSocket()
      logger.info( "initServerSocket()")

      software = "\# aprs4r #{APRS4R::APRS4RVersion}"
      @socket.puts( software + 0x0D.chr)

      begin
        Timeout::timeout( 5) do
          line = @socket.gets
          line.chomp! if line

          logger.debug( "start: #{line}")
        end
      rescue Exception => ex
        logger.warn( "initServerSocket::ex: #{ex}")
      end

      return
    end


    def reinitSocket()
      logger.info( "reinitSocket")

      # close socket (with timeout)
      begin
        Timeout::timeout( 5) do
          if ! @socket.nil?
            @socket.close
          end
        end
      rescue Exception => ex
        logger.warn( "reinitSocket::ex: #{ex}")
      end
      
      if @clientMode
        reinitClientSocket()
      else 
        reinitServerSocket()
      end

      return
    end
    
    
    def reinitClientSocket()
      logger.info( "reinitClientSocket")

      initClientSocket()
      
      return
    end


    def reinitServerSocket()
      logger.info( "reinitServerSocket")

      raise "client side down"

      return
    end


    def readAPRSMessage
      logger.info( "readAPRSMessage")

      data = readData

      message = parseAPRSMessage( data)

      logger.debug( "readAPRSMessage: message: #{message} #{message.nil?}")

      return message
    end

    
    def readData
      logger.info( "readData")

      line = nil

      begin
        Timeout::timeout( @timeout) do

          if @socket
            line = @socket.gets
          end

          # if nil, an exception is raised (and reinit is started)
          line.chomp!
          logger.debug( "readData::line: -#{line}-")

        end
      rescue Exception => ex
        logger.warn( "readData::ex: " + ex)
        reinitSocket()
      end until line !~ /^#/
        
        return line
    end
    

    def parseAPRSMessage( data)
      logger.info( "parseAPRSMessage( data)")
      
      logger.debug( "data: #{data}")

      if data.nil?
        return nil
      end
      
      # todo more robust message parsing 

      message = nil
      headerIndex = data.index( ':')
      
      if !headerIndex.nil?
        
        message = APRSMessage.new
        # header
        header = data[0...headerIndex]
        
        sourceIndex = header.index( '>')
        
        if sourceIndex != nil 
          message.source = APRSCall.parse( header[0...sourceIndex]).to_s
        end

        # path
        message.path = Array.new
        path = header[sourceIndex+1...header.length].split( ',')
        message.destination = APRSCall.parse( path[0]).to_s

        path.delete_at( 0)
        
        path.each{ |entry|
          call = APRSCall.parse( entry).to_s
          message.path << call
        }

        # body
        body = data[headerIndex+1...data.length]
        message.payload = body
      end
      return message
    end

    
    def writeAPRSMessage( message) 
      logger.info( "writeAPRSMessage( message)")
      
      data = serializeAPRSMessage( message)

      writeData( data)
      
      return
    end

    
    def serializeAPRSMessage( message)
      logger.info( "serializeAPRSMessage( message)")

      if message.nil?
        return nil
      end
      
      if message.source.nil? || message.destination.nil? 
        logger.warn( "NIL\n")
        return nil
      end

      data = String.new

      data = data + message.source.to_s + ">" + message.destination.to_s

      path = message.path
      
      for i in 0...path.length
        data = data + "," + path[i].to_s
      end

      data = data + ":" + message.payload

      return data
    end


    def writeData( data)
      logger.info( "writeData( data)")

      if data.nil?
        return
      end

      @mutex.synchronize do
        begin
          if @socket 
            logger.debug( "writeData::data: -#{data}-")
            @socket.puts( data + 0x0D.chr)
            @socket.flush
          end
        rescue Exception => ex
          logger.warn( "writeData::ex: " + ex)
          reinitSocket()
        end
      end

      return
    end

    
    def ISSocket.generatePassword( value)
      logger.info( "generatePassword( value)")

      callsign = APRSCall.parse( value)
      call = callsign.call

      if call.nil?
        return 
      end

      # init value
      value = 0x73E2
      i = 0

      while i < call.length
        value = value ^ (call[i].to_i << 8)
        value = value ^ (call[i+1].to_i)

        i += 2
      end

      return (value & 0x7FFF).to_s
    end

  end

end
