#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

# require 'aprs4r/AX25Socket'
# require 'aprs4r/ISSocket'
# require 'aprs4r/RawSocket'

require 'aprs4r/SocketRecvThread'
require 'aprs4r/SocketSendThread'


class SocketManager

  @@singleton = nil

  @@recvThreads = Hash.new
  @@sendThreads = Hash.new

  @@sockets = Hash.new
  @@listeners = Hash.new
  @@duplicates = Hash.new

  @@recvCount = Hash.new
  @@sendCount = Hash.new
  @@lastMessage = nil


  @@logger = Logger.get_logger( "SocketManager")


  def SocketManager.init( configuration)
    @@logger.info( "init( configuration)")

    sockets = configuration.devices

    if sockets.nil?
      return
    end

    sockets.each { |name, configuration|

      @@logger.debug( "configuration: #{configuration}")
      
      name = configuration.name
      type = configuration.type
      enable = configuration.enable

      if enable 
        @@logger.debug( "name: #{name}")
        @@logger.debug( "type: #{type}")
        @@logger.debug( "configuration: #{configuration}")
        
        socket = nil
        socket_class = "aprs4r" + "/" + type

        begin 
          require socket_class

          socket = eval( "#{type}.new( configuration)")
          
        rescue LoadError
          @@logger.warn( "unable to load socket #{type}")
          socket = nil
        end

        if socket 
          @@logger.debug( "socket: #{socket}")
        
          # start recv thread
          recvThread = SocketRecvThread.new( socket)
          @@recvThreads[name] = recvThread

          Thread.new { 
            recvThread.run
          }

          # start send thread
          sendThread = SocketSendThread.new( socket)
          @@sendThreads[name] = sendThread
          Thread.new { 
            sendThread.run
          }

          Thread.new { 
            run( name)
          }
        
          @@sockets[name] = socket
          @@listeners[name] = Array.new
          @@duplicates[name] = Hash.new
          
          @@recvCount[name] = 0
          @@sendCount[name] = 0
        end
      end
    }

    return
  end


  def SocketManager.run( name)
    @@logger.info( "run( name)")

    recvThread = @@recvThreads[name]

    while true 
      message = recvThread.recvAPRSMessage

      SocketManager.recvAPRSMessage( name, message)
    end

  end

  
  def SocketManager.addListener( name, listener)
    @@logger.info( "addListener( name, listener)")
    
    if @@sockets.key?( name) 
      listeners = @@listeners[name]
      listeners << listener
    end

    return
  end


  def SocketManager.removeListener( name, listener)
    @@logger.info( "removeListener( name, listener)")

    if @@sockets.key?( name)
      listeners = @@listeners[name]
      listeners.delete( listener)
    end

    return
  end

  
  def SocketManager.recvAPRSMessage( name, message)
    @@logger.info( "recvAPRSMessage( name, message)")

    if message.nil?
      return
    end

    if SocketManager.duplicate?( name, message)
      @@logger.warn( "recv: duplicate message dropped")
      return
    end

    @@recvCount[name] += 1
    @@lastMessage = message

    # deliver message to all registered listeners
    listeners = @@listeners[name]

    if ! listeners.nil?
      listeners.each{ |listener|
        listener.receiveAPRSMessage( name, message.clone)
      }
    end

    return
  end

  
  def SocketManager.sendAPRSMessage( name, message)
    @@logger.info( "sendAPRSMessage( name, message)")
    
    sendThread = @@sendThreads[name]

    if message.nil? || sendThread.nil?
      @@logger.warn( "message || thread: nil\n")
      return
    end

    if SocketManager.duplicate?( name, message)
      @@logger.warn( "send: duplicate message: #{message.source}")
      return
    end

    @@sendCount[name] += 1

    # send message and wakeup waiting thread
    sendThread.sendAPRSMessage( name, message)

    SocketManager.addDuplicate( name, message)

    return
  end


  def SocketManager.addDuplicate( name, message, now = Time.now)
    @@logger.info( "addDuplicate( name, message, now)")

    duplicatePeriod = 0

    socket = @@sockets[name]

    if !socket.nil?
      duplicatePeriod = socket.duplicatePeriod
    end

    duplicates = @@duplicates[name]

    if duplicatePeriod
      # remove old entries from duplicate list
      duplicates.each{ |key, timestamp|
        if now - timestamp > duplicatePeriod
          duplicates.delete( key)
        end
      }
    end
    
    # add message to duplicate list
    duplicates[message.duplicate_key] = now

    return
  end


  def SocketManager.duplicate?( name, message, now = Time.now)
    @@logger.info( "duplicate?( name, message, now)")
    
    duplicatePeriod = 0

    socket = @@sockets[name]

    if !socket.nil?
      duplicatePeriod = socket.duplicatePeriod
    end

    # duplicate check (if enabled)
    if duplicatePeriod 
      duplicates = @@duplicates[name]
      
      timestamp = duplicates[message.duplicate_key]
      
      # drop if timestamp exists and message is a duplicate
      if timestamp && (now - timestamp < duplicatePeriod)
        return true
      end
    end

    return false
  end

  
  def SocketManager.recv_count( name)
    return @@recvCount[name]
  end

  def SocketManager.send_count( name)
    return @@sendCount[name]
  end

  def SocketManager.lastMessage
    return @@lastMessage
  end

end
