#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'aprs4r/DigipeaterPluginConfiguration'
require 'aprs4r/Plugin'
require 'aprs4r/SocketManager'


class DigipeaterPlugin < Plugin

  @@RFONLY = "RFONLY"

  @@logger = Logger.get_logger( "DigipeaterPlugin")


  def initialize( configuration)
    @@logger.info( "initialize( configuration)")

    super( configuration)

    @device = configuration.device
    @call = configuration.call

    @mode = "fillin" 
    @mode = "wide" if configuration.mode == "wide"

    @aliases = Array.new
    if configuration.aliases
      configuration.aliases.each{ |entry|
        call = APRSCall.parse( entry)
        @aliases << call
      }
    end

    @rfonly = false
    @rfonly = configuration.rfonly if configuration.rfonly == true

    @reducePath = false
    @reducePath = configuration.reducePath if configuration.reducePath == true
    
    @permittedHops = configuration.permittedHops

    return
  end


  def registerListener
    @@logger.info( "registerListener")

    if @enable
      SocketManager.addListener( @device, self)
    end

    return
  end


  def unregisterListener
    @@logger.info( "registerListener")

    if @enable
      SocketManager.removeListener( @device, self)
    end

    return
  end


  def start
    @@logger.info( "start")

    if ! @enable 
      return
    end

    registerListener

    return
  end


  def receiveAPRSMessage( name, message)
    @@logger.info( "receiveAPRSMessage( name, message)")

    if message.nil?
      @@logger.warn( "message: nil\n")
      return
    end

    message = do_digipeating( name, message)

    if message 
      @@logger.debug( "send message: #{message}")
      SocketManager.sendAPRSMessage( @device, message)
    end
      
    return
  end


  def do_digipeating( name, message)
    @@logger.info( "do_digipeating( name, message)")

    if message.nil?
      @@logger.warn( "no message")
      return
    end

    if message.path.nil? || message.path.empty?
      @@logger.warn( "no path")
      return
    end

    # translate path
    message = translate_path( message)
    @@logger.debug( "do_digipeating: path translate: #{message}")

    # optimize path 
    message = optimize_path( message)
    @@logger.debug( "do_digipeating: path optimize: #{message}")

    path = message.path

    # reduce path (if enabled)
    if @reducePath
      rfonly = false
      path.each{ |entry|
        rfonly = true if entry == @@RFONLY
      }

      if (@rfonly == false) || ((@rfonly == true) && (rfonly == false))
        message = reduce_path( message)
        @@logger.debug( "do_digipeating: path reduce: #{message}")
      end
    end

    path = message.path

    # no path, no digipeating
    if message.path.nil? || message.path.empty?
      @@logger.debug( "no path")
      return
    end

    # search for last repeated flag
    repeated_index = 0
    for i in 0...path.length
      entry = APRSCall.parse( path[i])

      repeated_index = i+1 if entry.repeated
    end

    repeat = false
    wide_keyword = /WIDE1/
    wide_keyword = /(TRACE|WIDE)[1-7]/ if @mode == "wide"

    
    # search for keyword
    for i in repeated_index...path.length
      entry = APRSCall.parse( path[i])

      # check for wide keyword
      if entry.call =~ wide_keyword && entry.ssid > 0 
        repeat = true

        entry.decrease_ttl
        digi = APRSCall.parse( @call)

        if entry.ssid == 0 
          entry.repeated = true
        else
          digi.repeated = true 
        end

        path[i] = entry.to_s
        path.insert( i, digi.to_s)

        message.path = path

        # remove other repeated flags
        for j in 0...repeated_index
          entry = APRSCall.parse( path[j])
          entry.repeated = false
          path[j] = entry.to_s
        end
        
        return message
      end

      # check for aliases (and replace and disable digipeater call)
      if @aliases.include?( entry)
        # remove other repeated flags
        for j in 0...repeated_index
          entry = APRSCall.parse( path[j])
          entry.repeated = false
          path[j] = entry.to_s
        end

        call = APRSCall.parse( @call)
        call.repeated = true 
        path[i] = call.to_s

        return message
      end

    end
    
    return
  end

  
  def translate_path( message)
    @@logger.info( "translate_path( message)")
    
    if message.nil?
      @@logger.warn( "message: NIL")
      return
    end

    path = message.path

    if path.nil?
      @@logger.warn( "path: NIL")
      return
    end

    # search for last repeated flag
    repeated_index = 0
    for i in 0...path.length
      entry = APRSCall.parse( path[i])

      repeated_index = i+1 if entry.repeated
    end

    behind_relay = false
    behind_wide1 = false
    behind_trace = false

    for i in repeated_index...path.length
      entry = path[i]
      call = nil

      if entry == "RELAY"
        if behind_relay
          call = APRSCall.parse( "WIDE2-1")
        else
          call = APRSCall.parse( "WIDE1-1")
          behind_relay = true
        end
      elsif entry == "WIDE" 
        call = APRSCall.parse( "WIDE2-1")
      elsif entry == "WIDE1-1"
        if behind_wide1
          call = APRSCall.parse( "WIDE2-1")
        else
          bedind_wide1 = true
        end
      elsif entry =~ /WIDE[0-9]/
        if behind_trace
          call = "TRACE" + entry[4...7]
        end
      elsif entry == "TRACE" 
        call = APRSCall.parse( "TRACE2-1")
        behind_trace = true
      elsif entry =~ /TRACE[0-9]/ && @translateTrace
        behind_trace = true 
      end

      path[i] = call.to_s if call
    end

    return message
  end
  

  def optimize_path( message)
    @@logger.debug( "optimizePath( message)")

    if message.nil?
      @@logger.warn( "message: NIL")
      return
    end

    path = message.path
    path_new = Array.new

    if path.nil?
      @@logger.warn( "path: NIL")
      return
    end

    # search for last repeated flag
    repeated_index = 0
    for i in 0...path.length
      entry = APRSCall.parse( path[i])

      repeated_index = i+1 if entry.repeated
    end

    @@logger.debug( "optimize_path: repeated_index: #{repeated_index}")

    for i in 0...repeated_index 
      @@logger.debug( "optimize_path: adding entry: #{path[i]}")
      path_new << path[i]
    end

    last_wide_ssid = 0
    last_trace_ssid = 0
    for i in repeated_index...path.length
      entry = path[i]
      @@logger.debug( "entry: #{entry}")
      call = APRSCall.parse( entry)
      current_ssid = call.ssid

      if call.call =~ /WIDE[2-9]/ && call.ssid > 0 
        if last_wide_ssid + current_ssid <= 7
          last_wide_ssid += current_ssid
        else 
          wide_new = "WIDE" + last_wide_ssid.to_s + "-" + last_wide_ssid.to_s
          wide_new = "WIDE2-1" if wide_new == "WIDE1-1"
          path_new << wide_new
          last_wide_ssid = current_ssid
        end
      elsif call.call =~ /TRACE[2-9]/ && call.ssid > 0
        if last_trace_ssid + current_ssid <= 7
          last_trace_ssid += current_ssid
        else 
          trace_new = "TRACE" + last_trace_ssid.to_s + "-" + last_trace_ssid.to_s
          trace_new = "TRACE2-1" if trace_new == "TRACE1-1"
          path_new << trace_new
          last_trace_ssid = current_ssid
        end
      else     
        @@logger.debug( "optimize_path: adding entry: #{call}")
        path_new << entry
      end

    end

    if last_wide_ssid > 0 
      wide_new = "WIDE" + last_wide_ssid.to_s + "-" + last_wide_ssid.to_s
      wide_new = "WIDE2-1" if wide_new == "WIDE1-1"
      path_new << wide_new
    end

    if last_trace_ssid > 0 
      trace_new = "TRACE" + last_trace_ssid.to_s + "-" + last_trace_ssid.to_s
      trace_new = "TRACE2-1" if trace_new == "TRACE1-1"
      path_new << trace_new
    end
      

    message.path = path_new

    return message
  end


  def reduce_path( message) 
    @@logger.info( "reduce_path( message)")
    
    if message.nil?
      @@logger.warn( "message: NIL")
      return
    end

    path = message.path
    path_new = Array.new

    if path.nil?
      @@logger.warn( "path: NIL")
      return
    end

    if @permittedHops.nil?
      @@logger.warn( "permittedHops not set")
      return
    end

    ssid_sum = 0

    # search for last repeated flag
    repeated_index = 0
    for i in 0...path.length
      entry = APRSCall.parse( path[i])

      repeated_index = i+1 if entry.repeated
      
      if entry.call =~ /WIDE[2-9]/ || entry.call =~ /TRACE[2-9]/
        ssid_sum += entry.ssid
      end
    end

    @@logger.debug( "optimize_path: repeated_index: #{repeated_index}")

    for i in 0...repeated_index 
      @@logger.debug( "optimize_path: adding entry: #{path[i]}")
      path_new << path[i]
    end

    @@logger.debug( "ssid_sum: #{ssid_sum}")
    @@logger.debug( "permittedHops: #{@permittedHops}")

    for i in repeated_index...path.length
      entry = path[i]
      call = APRSCall.parse( entry)

      if entry =~ /WIDE[2-9]/ || entry =~ /TRACE[2-9]/
        ssid = call.ssid

        if ssid_sum > @permittedHops

          @@logger.debug( "in if: ssid_sum: #{ssid_sum}, permittedHops: #{@permittedHops}")
          
          decrement = [ ssid_sum - @permittedHops, ssid].min

          call.ssid -= decrement
          ssid_sum -= decrement

          if call.ssid != 0 
            path_new << call.call[0...call.call.length-1] + call.ssid.to_s + "-" + call.ssid.to_s
          end
        else
          path_new << entry
        end
        
      else
        path_new << entry
      end
    end

    message.path = path_new

    return message
  end

end
