#!/usr/bin/ruby
#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'include'

require 'test/unit'

require 'aprs4r/APRSCall'


class APRSCallTest < Test::Unit::TestCase

  
  def test_new_none
    call = APRSCall.new
    assert_equal( "", call.call)
    assert_equal( 0, call.ssid)
    assert_equal( false, call.repeated)
  end

  def test_new_call
    call = APRSCall.new( "ABCDEF")
    assert_equal( "ABCDEF", call.call)
    assert_equal( 0, call.ssid)
    assert_equal( false, call.repeated)
  end

  def test_new_call_ssid
    call = APRSCall.new( "ABCDEF", 8)
    assert_equal( "ABCDEF", call.call)
    assert_equal( 8, call.ssid)
    assert_equal( false, call.repeated)
  end

  def test_new_call_ssid_repeated
    call = APRSCall.new( "ABCDEF", 8, true)
    assert_equal( "ABCDEF", call.call)
    assert_equal( 8, call.ssid)
    assert_equal( true, call.repeated)
  end


  def test_to_s_call
    call = APRSCall.new( "ABCDEF")
    assert_equal( "ABCDEF", call.to_s)
  end


  def test_to_s_ssid
    call = APRSCall.new( "ABCDEF", 5)
    assert_equal( "ABCDEF-5", call.to_s)
  end

  def test_to_s_repeated
    call = APRSCall.new( "ABCDEF", 0, true)
    assert_equal( "ABCDEF*", call.to_s)
  end

  def test_to_s_full
    call = APRSCall.new( "ABCDEF", 5, true)
    assert_equal( "ABCDEF-5*", call.to_s)
  end


  def test_parse_call
    call = APRSCall.parse( "ABCDEF")
    assert_equal( "ABCDEF", call.to_s)
  end

  def test_parse_ssid
    call = APRSCall.parse( "ABCDEF-4")
    assert_equal( "ABCDEF-4", call.to_s)

    call = APRSCall.parse( "ABCDEF-0")
    assert_equal( "ABCDEF", call.to_s)
  end
  
  def test_parse_repeated
    call = APRSCall.parse( "ABCDEF*")
    assert_equal( "ABCDEF*", call.to_s)
  end

  def test_parse_full
    call = APRSCall.parse( "ABCDEF-4*")
    assert_equal( "ABCDEF-4*", call.to_s)

    call = APRSCall.parse( "ABCDEF-0*")
    assert_equal( "ABCDEF*", call.to_s)
  end


  def test_decrease_ttl_none
    call = APRSCall.new( "ABCDEF")
    call.decrease_ttl
    assert_equal( "ABCDEF", call.to_s)
  end
    

  def test_decrease_ttl_simple
    call = APRSCall.new( "WIDE5", 5)
    call.decrease_ttl
    assert_equal( "WIDE5-4", call.to_s)
  end
    
  def test_decrease_ttl_special
    call = APRSCall.new( "WIDE2", 1)
    assert_equal( "WIDE2-1", call.to_s)
    call.decrease_ttl
    assert_equal( "WIDE2", call.to_s)
  end

  
  def test_clone
    call1 = APRSCall.new( "ABCDEF", 5, false)
    call2 = call1.clone

    call2.call = "BCDEFG"
    call2.ssid = 4
    call2.repeated = true

    assert_equal( "ABCDEF-5", call1.to_s)
    assert_equal( "BCDEFG-4*", call2.to_s)
    assert_not_equal( call1.to_s, call2.to_s)
  end

end
  
