#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'serialport'
require 'thread'

require 'aprs4r/APRSMessage'
require 'aprs4r/WeatherPlugin'
require 'aprs4r/WMRWeatherMessage'
require 'aprs4r/WMRWeatherConfiguration'


class WMRWeatherPlugin < WeatherPlugin

  @@logger = Logger.get_logger( "WMRWeatherPlugin")

  
  def initialize( configuration)
    @@logger.info( "initialize( configuration)")

    super( configuration)

    @file = configuration.file

    # @port = File.new( @file, "r")
    @port = SerialPort.new( @file, 9600, 8, 1, SerialPort::NONE)
    @port.nonblock = true 
    @port.sync = true
    @port.flow_control = SerialPort::HARD

    @wind_direction = nil
    @wind_avg_speed = nil
    @wind_gust_speed = nil
    @rain_current = nil
    @rain_total = nil
    @rain_yesterday = nil
    @temperature = nil
    @humidity = nil
    @dewpoint = nil

    Thread.new{ runWMRThread }

    return
  end


  def createWeatherAPRSMessage
    @@logger.info( "createWeatherAPRSMessage")

    # copy default message
    message = @message.clone

    weather = String.new

    # write wind data
    wind_direction = "..."
    wind_direction = sprintf( "%3.3i", @wind_direction) if @wind_direction

    wind_avg_speed = "..."
    wind_avg_speed = sprintf( "%3.3i", @wind_avg_speed) if @wind_avg_speed

    wind_gust_speed = "g..."
    wind_gust_speed = sprintf( "g%3.3i", @wind_gust_speed) if @wind_gust_speed

    # temperature
    temperature = "t..."
    temperature = sprintf( "t%3.3i", @temperature) if @temperature

    # rain
    rain_current = "r..."
    rain_current = sprintf( "r%3.3i", @rain_current) if @rain_current
    
    rain_total = "p..."
    rain_total = sprintf( "p%3.3i", @rain_total) if @rain_total

    rain_yesterday = "R..."
    rain_yesterday = sprintf( "R%3.3i", @rain_total) if @rain_yesterday
    
    # humidity/pressure
    humidity = "h.."
    humidity = sprintf( "h%2.2i", @humidity) if @humidity

    pressure = "b....."
    pressure = sprintf( "b%5.5i", @pressure * 10) if @pressure

    weather += wind_direction + "/" + wind_avg_speed + wind_gust_speed
    weather += temperature + rain_current + rain_total + rain_yesterday
    weather += humidity + pressure

    @@logger.warn( "weather: #{weather}")

    message.payload += weather

    @@logger.warn( "message: #{message.to_s}")

    return message
  end


  def runWMRThread
    @@logger.info( "runWMRThread")
    
    while true 
      readWeatherData
    end

    return
  end


  def readWeatherData
    @@logger.info( "readWeatherData")

    record = readRecord

    if record 
      if record.has_wind?
        @@logger.info( "wind: direction: #{record.wind_direction}, avg: #{record.wind_avg_speed}, gust: #{record.wind_gust_speed}")
        @wind_direction = record.wind_direction
        @wind_avg_speed = record.wind_avg_speed
        @wind_gust_speed = record.wind_gust_speed
      end
      
      if record.has_rain?
        @@logger.info( "rain: current: #{record.rain_current}, total: #{record.rain_total}, yesterday: #{record.rain_yesterday}")
        @rain_current = record.rain_current
        @rain_total = record.rain_total
        @rain_yesterday = record.rain_yesterday
      end
      
      if record.has_thermo?
        @@logger.info( "thermo: temperature: #{record.temperature}, humidity: #{record.humidity}, dewpoint: #{record.dewpoint}")
        @temperature = record.temperature
        @humidity = record.humidity
        @dewpoint = record.dewpoint
      end

      if record.has_baro?
        @@logger.info( "baro: pressure: #{record.pressure_relative}")
        @pressure = record.pressure_relative
      end
    else 
      @@logger.warn( "empty record received")
    end
      
    return
  end


  def readRecord
    @@logger.info( "readRecord")

    type = 0x10
    payload = String.new
    checksum = 0

    # read until HEADER
    value = @port.getc
    while value != WMRWeatherMessage.WMR_HEADER
      print "value: #{value.chr.to_i}\n"
      value = @port.getc
    end

    # read all HEADER
    while value == WMRWeatherMessage.WMR_HEADER
      value = @port.getc
    end

    type = value 
    @@logger.debug( "type: #{type}")

    length = WMRWeatherMessage.message_length( type)
    @@logger.debug( "length: #{length}")
    
    payload = Array.new
    if length != nil && length > 4
      (length-4).times { 
        value = @port.getc
        @@logger.debug( "payload byte: #{value}")
        payload << value
      }
    end

    checksum = @port.getc
    @@logger.debug( "checksum: #{checksum}")

    # create wmr weather message
    record = WMRWeatherMessage.new( type, payload, checksum)
    
    if record.is_valid? 
      @@logger.debug( "message is valid")
      return record
    end
    
    @@logger.warn( "bad checksum, returning nil")

    return nil
  end
end
