#
#  APRS4R - a ruby based aprs gateway/digipeater
#  Copyright (C) 2006 by Michael Conrad <do5mc@friggleware.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
#

require 'Logger'

require 'aprs4r/APRSCall'


class APRSMessage

  @@APRS_PATH_RELAY = "RELAY"
  @@APRS_PATH_WIDE  = "WIDE"
  @@APRS_PATH_TRACE = "TRACE"

  @@APRS_POSITION = { '!' => true, '=' => true, '/' => true, '@' => true}
  @@APRS_POSITION_LATITUDE_INDEX = { '!' => 1, '=' => 1, '/' => 8, '@' => 8}
  @@APRS_POSITION_LONGITUDE_INDEX = { '!' => 10, '=' => 10, '/' => 17, '@' => 17}

  @@APRS_SYMBOL_TABLE_INDEX = { '!' => 9, '=' => 9, '/' => 16, '@' => 16 }
  @@APRS_SYMBOL_CODE_INDEX = { '!' => 19, '=' => 19, '/' => 26, '@' => 26 }

  @@APRS_EXTENSION_INDEX = { '!' => 20, '=' => 20, '/' => 27, '@' => 27 }
  @@APRS_EXTENSION_LENGTH = 7

  @@APRS_WEATHER_INDEX = { '_' => 10, '!' => 20, '=' => 20, '/' => 27, '@' => 27 } 
  @@APRS_WEATHER_LENGTH = 0

  @@APRS_COMMENT_INDEX = { '!' => 20, '=' => 20, '/' => 27, '@' => 27 }

  @@APRS_SYMBOL_DIGIPEATER = "#"
  @@APRS_SYMBOL_GATEWAY = "&"
  @@APRS_SYMBOL_WEATHER = "_"
  @@APRS_SYMBOL_HOUSE = "-"
  @@APRS_SYMBOL_YACHT = "Y"
  @@APRS_SYMBOL_BOAT = "s"
  @@APRS_SYMBOL_CAR = ">"

  @@APRS_SYMBOL_CODE_NAME = { 
    @@APRS_SYMBOL_DIGIPEATER => "digipeater",
    @@APRS_SYMBOL_GATEWAY => "gateway",
    @@APRS_SYMBOL_WEATHER => "weather", 
    @@APRS_SYMBOL_HOUSE => "house", 
    @@APRS_SYMBOL_YACHT => "yacht",
    @@APRS_SYMBOL_BOAT => "boat",
    @@APRS_SYMBOL_CAR => "car"
  }

  @@APRS_PHG_POWER = [0, 1, 4, 9, 16, 25, 36, 49, 64, 81]
  @@APRS_PHG_HEIGHT = [10, 20, 40, 80, 160, 320, 640, 1280, 2560, 5120]
  @@APRS_PHG_GAIN = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
  @@APRS_PHG_DIRECTION = ["omni", "45/NE", "90/E", "135/SE", "180/S", "225/SW", "270/W", "315/NW", "360/N"]
    
    
  attr_reader :source, :destination, :path, :payload
  attr_writer :source, :destination, :path, :payload


  @@logger = Logger.get_logger( 'APRSMessage')


  def initalize( source = APRSCall.new, destination = APRSCall.new, path = Array.new, payload = nil)
    @@logger.info( "initialize( source, destination, path, payload)")
    
    @source = source
    @destination = destination
    @path = path
    @payload = payload

    @type = nil
    
    return
  end


  def clone
    @@logger.info( "clone")
    
    message = APRSMessage.new

    message.source = @source.clone
    message.destination = @destination.clone

    path = Array.new
    @path.each{ |entry| path << entry.clone} if @path
    message.path = path

    message.payload = String.new( @payload)

    return message
  end


  def duplicate_key
    @@logger.info( "duplicate_key")
    
    key = String.new

    # ignore path for duplicate detection
    key << source.to_s << ">" << destination.to_s
    key << ":" << payload

    return key
  end


  def to_yaml_properties
    %w{@source @destination @path @payload}
  end


  def type
    type = nil

    if payload.length > 0 
      type = payload[0].chr
    end

    return type
  end


  def has_symbol?
    @@logger.info( "has_symbol?")

    return @@APRS_SYMBOL_TABLE_INDEX.has_key?( type)
  end


  def symbol_table
    @@logger.info( "symbol_table")

    if payload.length > 0 
      index = @@APRS_SYMBOL_TABLE_INDEX[type]

      if index && index < payload.length
        return payload[index].chr
      end
    end

    return nil
  end


  def symbol_code
    @@logger.info( "symbol_code")

    if payload.length > 0 
      index = @@APRS_SYMBOL_CODE_INDEX[type]

      if index && index < payload.length
        return payload[index].chr
      end
    end
    
    return nil
  end

  
  def symbol_name
    @@logger.info( "symbol_name")

    code = symbol_code
    name = @@APRS_SYMBOL_CODE_NAME[code]

    return name
  end


  def has_position?
    @@logger.info( "has_position?")

    if ! @@APRS_POSITION.has_key?( type) 
      return false
    end

    position_match = /[\d\ ]{4}\.[\d\ ]{2}[NS].[\d\ ]{5}\.[\d\ ]{2}[EW]/

    if payload =~ position_match
      return true
    end

    return false
  end
  

  def latitude
    @@logger.info( "latitude")

    latitude = 0.0

    if has_position?
      index = @@APRS_POSITION_LATITUDE_INDEX[type]
      length = 8

      if index + length < payload.length
        latitude = APRSMessage.degree2decimal( payload[index...index+length])
      end
    end

    return latitude
  end


  def longitude
    @@logger.info( "longitude")

    longitude = 0.0

    if has_position?
      index = @@APRS_POSITION_LONGITUDE_INDEX[type]
      length = 9

      if index + length < payload.length
        longitude = APRSMessage.degree2decimal( payload[index...index+length])
      end
    end

    return longitude
  end

  
  def has_extension?
    @@logger.info( "has_extension?")

    course = has_course?
    @@logger.info( "course: #{course}")

    phg = has_phg?
    @@logger.info( "phg: #{phg}")

    return course || phg
  end


  def has_comment?
    @@logger.info( "has_comment?")

    if has_weather? 
      return false
    end

    return @@APRS_COMMENT_INDEX.has_key?( type)
  end


  def comment
    @@logger.info( "comment")
    
    comment = String.new
    index = @@APRS_COMMENT_INDEX[type]

    if has_extension?
      index += @@APRS_EXTENSION_LENGTH
    end

    if has_comment? && payload.length > index
      comment = payload[index...payload.length]
    end

    return comment
  end


  def has_status?
    @@logger.info( "has_status?")

    return type == '>'
  end


  def status
    @@logger.info( "status")

    status = String.new

    if has_status? && payload.length > 1
      status = payload[1...payload.length]
    end
    
    return status
  end

  
  def has_weather?
    # NOTE: compare type id with symbol (equal for weather stations)

    if type != @@APRS_SYMBOL_WEATHER && symbol_code != @@APRS_SYMBOL_WEATHER
      return false
    end

    return type == @@APRS_SYMBOL_WEATHER || symbol_code == @@APRS_SYMBOL_WEATHER
  end

  
  def weather_wind_direction
    direction = 0
    index = @@APRS_WEATHER_INDEX[type]

    if index && has_weather? && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
      if data && data.length > 0 
        direction = data[0][0..2].to_i
      end
      # direction = payload[index..index+3].to_i
    end

    return direction
  end


  def weather_wind_speed
    speed = 0
    index = @@APRS_WEATHER_INDEX[type]

    if index && has_weather? && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
      if data && data.length > 0 
        speed = data[0][4..6].to_i
      end
      # speed = payload[index+4..index+6].to_i
    end

    return speed
  end


  def weather_wind_gust
    speed = 0

    index = @@APRS_WEATHER_INDEX[type]

    if has_weather? && index && payload.lengt >= index
      data = payload[index..payload.length].to_s.scan( /g\d\d\d/)
      if data && data.length > 0 
        speed = data[0][1..3].to_i
      end
    end

    return speed
  end


  def weather_temperature
    temperature = 0

    index = @@APRS_WEATHER_INDEX[type]

    if has_weather? && index && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /t\d\d\d/)
      if data && data.length > 0 
        temperature = data[0][1..3].to_i
      end
      # temperature = payload[index+12..index+14].to_i
    end

    return temperature
  end


  def weather_rain
    rain = 0 

    index = @@APRS_WEATHER_INDEX[type]

    if has_weather? && index && payload.length >= index 
      data = payload[index..payload.length].to_s.scan( /r\d\d\d/)
      if data && data.length > 0 
        rain = data[0][1..3].to_i
      end
    end

    return rain
  end


  def weather_humidity
    humidity = 0

    index = @@APRS_WEATHER_INDEX[type]

    if has_weather? && index && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /h\d\d/)
      if data && data.length > 0 
        humidity = data[0][1..2].to_i
      end
    end

    return humidity
  end


  def weather_pressure
    pressure = 0 
    
    index = @@APRS_WEATHER_INDEX[type]

    if has_weather? && index && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /b\d\d\d\d\d/)
      if data && data.length > 0
        pressure = data[0][1..5].to_f / 10.0
      end
    end

    return pressure
  end


  def has_course?
    @@logger.info( "has_course?")

    index = @@APRS_COMMENT_INDEX[type]

    if symbol_code == @@APRS_SYMBOL_WEATHER
      return false
    end

    if index && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
      if data && data.length > 0 
        return true 
      end
    end

    return false
  end


  def course_direction
    direction = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_course? && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
      if data && data.length > 0 
        direction = data[0][0..2].to_i
      end
      # direction = payload[index..index+3].to_i
    end

    return direction
  end


  def course_speed
    speed = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_course? && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /\d\d\d\/\d\d\d/)
      if data && data.length > 0 
        speed = data[0][4..6].to_i
      end
      # speed = payload[index+4..index+6].to_i
    end

    return speed
  end


  def has_phg?
    @@logger.info( "has_phg?")

    index = @@APRS_COMMENT_INDEX[type]

    if index && payload.length >= index
      data = payload[index..payload.length].to_s.scan( /PHG\d\d\d\d/)
      if data && data.length > 0 
        return true 
      end
    end

    return false
  end


  def phg_power
    power = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_phg? && payload.length >= (index+7)
      value = payload[index+3].chr.to_i

      if value >= 0 && value < 10 
        power = @@APRS_PHG_POWER[value]
      end
    end

    return power
  end


  def phg_height
    height = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_phg? && payload.length >= (index+7)
      value = payload[index+4].chr.to_i

      if value >= 0 && value < 10 
        height = @@APRS_PHG_HEIGHT[value]
      end
    end

    return height
  end


  def phg_gain
    gain = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_phg? && payload.length >= (index+7)
      value = payload[index+5].chr.to_i

      if value >= 0 && value < 10 
        gain = @@APRS_PHG_GAIN[value]
      end
    end

    return gain
  end


  def phg_direction
    direction = 0
    index = @@APRS_COMMENT_INDEX[type]

    if index && has_phg? && payload.length >= (index+7)
      value = payload[index+6].chr.to_i

      if value >= 0 && value < 10 
        direction = @@APRS_PHG_DIRECTION[value]
      end
    end

    return direction
  end

  
  def is_message?
    return type == ':'
  end

  
  def is_query?
    return type == '?'
  end


  def to_s
    buffer = String.new

    # buffer << "APRSMessage: "
    buffer << @source.to_s << " -> "
    buffer << @destination.to_s << " via "
    
    path = String.new
    for i in 0...@path.length

      if i != 0 
        path = path + ", "
      end

      path = path + @path[i].to_s
    end
    path = path

    buffer << "[" << path << "]: "
    buffer << "(" << @payload << ")"

    return buffer.to_s
  end

  
  def APRSMessage.degree2decimal( position)
    @@logger.info( "degree2decimal")

    value = 0.0

    if position.nil? || position.length < 8
      return value
    end

    case position[position.length-1].chr
    when 'N'
      value = position[0...2].to_f + (position[2...4].to_f + position[5...7].to_f / 100.0) / 60.0
      
    when 'S'
      value = position[0...2].to_f + (position[2...4].to_f + position[5...7].to_f / 100.0) / 60.0
      value *= -1.0

    when 'E'
      value = position[0...3].to_f + (position[3...5].to_f + position[6...8].to_f / 100.0) / 60.0

    when 'W'
      value = position[0...3].to_f + (position[3...5].to_f + position[6...8].to_f / 100.0) / 60.0
      value *= -1.0

    end

    return value
  end


  def APRSMessage.APRS_PATH_RELAY
    return @@APRS_PATH_RELAY
  end

  def APRSMessage.APRS_PATH_WIDE
    return @@APRS_PATH_WIDE
  end

  def APRSMessage.APRS_PATH_TRACE
    return @@APRS_PATH_TRACE
  end

  def APRSMessage.attributes 
    
    attributes = [
                  ConfigurationAttribute.new( "source", 01, "text", "Quelle", 10),
                  ConfigurationAttribute.new( "destination", 02, "text", "Ziel", 10), 
                  ConfigurationAttribute.new( "path", 03, "text", "Pfad", 10),
                  ConfigurationAttribute.new( "payload", 04, "text", "Nutzdaten", 40)
                 ]

    return attributes
  end

end
